;  ------ Power Amplifier Microcomputer Program ------
;  ------ Processor: AT89C51 24PC --------------------

;#DEFINE AUTO_OFF		;auto off timer enable
#DEFINE DISP_FADE		;display fade enable
;#DEFINE RESTORE_DM		;restore display mode enable

#INCLUDE "LIBREG.ASM"	;8051 SFR set

CLK_KHZ	= 12000		;OSC frequency, KHZ
RTC_MS	= 20		;System clock , MS

RTCV	= -((CLK_KHZ*RTC_MS)/12)

;  ------ Macros ------

#DEFINE LO(XXX) XXX & 0FFH
#DEFINE HI(XXX) (XXX >> 8) & 0FFH

;  ------ Constantes ------

;General:
STACK	.EQU 057H		;stack location
SYS	.EQU 00AH		;RC5 system
V1S	.EQU 50		;program timer RTPCS value 1S/20mS=50
V1M	.EQU 60		;program timer RTPCM value 1MIN/1S=60
DISFTMV	.EQU 20H		;display fade rate
SLEW	.EQU 08H		;display slew rate

;Temperature:
TFLIM	.EQU 598		;598/10=59.8 Deg. C temp. fail limit
TILIM	.EQU 550		;550/10=55.0 Deg. C temp. ind. limit
THLIM	.EQU 450		;450/10=45.0 Deg. C temp. hi-fan  limit
TLLIM	.EQU 350		;350/10=35.0 Deg. C temp. fan  limit

;Power:
PALIM	.EQU 0005		;0005/10=000.5W power auto off limit
PNLIM	.EQU 0020		;0020/10=002.0W power fan off limit

;Voltage:
UHLIM	.EQU 38		;38V U high limit
ULLIM	.EQU 20		;20V U low  limit

;Time x1min:
AUFTMV	.EQU 5		;5min	auto off timer delay value

;Time x1S:
FANTMV	.EQU 10		;10S	fan timer value
UWTDEL	.EQU 5		;5S	U wait delay

;Time x20mS:
KEYID	.EQU 40		;800mS	key hold initial delay
KEYDD	.EQU 18		;360mS	key hold autorepeat delay
STBHTMV	.EQU 40		;800mS	standby hold delay
TMRHTMV	.EQU 40		;800mS	timer hold delay
DISPTMV	.EQU 16		;320mS	display update rate
PHLDTMV	.EQU 70		;1400mS	peak power hold time
PDRATE	.EQU 1		;20mS	power dec rate
TEMPTMV	.EQU 50		;1000mS	thermometer ready time
STRDEL	.EQU 50		;1000mS	standby reverse delay
INIDEL	.EQU 50		;1000mS	initial ST  to OP  delay
EXPDEL	.EQU 25		;500mS	EXP to DYS delay
DYSDEL	.EQU 50		;1000mS	DYS to RP1 delay
RP1DEL	.EQU 15		;300mS	RP1 to UOK delay
RP2DEL	.EQU 25		;500mS	UOK to RP2 min delay
LDRDEL	.EQU 100		;2000mS	RP2 to LDR delay
LDFDEL	.EQU 10		;200mS	LDR to RP2 delay
P2FDEL	.EQU 10		;200mS	RP2 to RP1 delay
P1FDEL	.EQU 25		;500mS	RP1 to MNP delay
EXFDEL	.EQU 10		;200mS	MNP to EXP delay
FLDEL	.EQU 5		;100mS	fail process delay

;  ------ Ports ------

RETL	.EQU INT0		;keyboard return line
SER	.EQU INT1		;RC-5 data line
BPR	.EQU T0		;beeper line
WD	.EQU T1		;watchdog line
OWP1	.EQU P1.0		;1-Wire port 1
OWP2	.EQU P1.1		;1-Wire port 2
START	.EQU P1.2		;START ADC line
SDATA	.EQU P1.3		;SDATA ADC line
SCLK	.EQU P1.4		;SCLK  ADC line
AX0	.EQU P1.5		;analog MUX address A0
AX1	.EQU P1.6		;analog MUX address A1
AX2	.EQU P1.7		;analog MUX address A2
FAN	.EQU P2.0		;fan motor
RP1	.EQU P2.1		;power relay 1
RP2	.EQU P2.2		;power relay 2
REX	.EQU P2.3		;external power relay
PEN	.EQU P2.4		;power enable
ENB	.EQU P2.5		;protection relay enable
IOF	.EQU P2.6		;protection current fail
DCF	.EQU P2.7		;protection DC fail
OKL	.EQU TXD		;protection left  OK
OKR	.EQU RXD		;protection right OK

;  ------ Variables ------

;  Bit addressing memory

RTPC	.EQU 020H		;Real time program counter
T040M	.EQU M_20H.0	;40mS  period bit
T080M	.EQU M_20H.1	;80mS  period bit
T160M	.EQU M_20H.2	;160mS period bit
T320M	.EQU M_20H.3	;320mS period bit
T640M	.EQU M_20H.4	;640mS period bit
T1S28	.EQU M_20H.5	;1.28S period bit
T2S56	.EQU M_20H.6	;2.56S period bit
T5S12	.EQU M_20H.7	;5.12S period bit

BLINK	.EQU T640M	;blinking period

RTPCS	.EQU 021H		;Real time program counter (1S   part)
RTPCM	.EQU 022H		;Real time program counter (1MIN part)

LEDS1	.EQU 023H
LEDEXT	.EQU M_23H.0	;external power LED
LEDDGR	.EQU M_23H.1	;degree indication LED
POINT	.EQU M_23H.2	;decimal point
LEDWTT	.EQU M_23H.3	;watt indication LED

LEDS2	.EQU 024H
LEDTME	.EQU M_24H.0	;time indication LED
LEDLFL	.EQU M_24H.1	;left channel fail LED
LEDMNP	.EQU M_24H.2	;main power LED
LEDTMR	.EQU M_24H.3	;timer LED
LEDOPR	.EQU M_24H.4	;operate LED
LEDPKP	.EQU M_24H.5	;peak power indication LED
LEDLDR	.EQU M_24H.6	;load relay LED
LEDRFL	.EQU M_24H.7	;right channel fail LED

PRESS	.EQU M_25H.0	;press flag for local keyboard
OFF_EN	.EQU M_25H.1	;off enable flag
EX1_EN	.EQU M_25H.2	;EX1 enable flag
EXTLVE	.EQU M_25H.3	;EXT leave flag
KEYHE	.EQU M_25H.4	;key hold enable flag

FAIL_L	.EQU 026H		;left channel fail flags
IF_L	.EQU M_26H.0	;I out fail
DF_L	.EQU M_26H.1	;DC out fail
UF_L	.EQU M_26H.2	;U supply fail
PF_L	.EQU M_26H.3	;protection fail
TF_L	.EQU M_26H.4	;heatsink temperature fail

FAIL_R	.EQU 027H		;right channel fail flags
IF_R	.EQU M_27H.0	;I out fail
DF_R	.EQU M_27H.1	;DC out fail
UF_R	.EQU M_27H.2	;U supply fail
PF_R	.EQU M_27H.3	;protection fail
TF_R	.EQU M_27H.4	;heatsink temperature fail

STATES	.EQU 028H		;standby/operate/fail processes states
F_EXT	.EQU M_28H.0	;external power flag
F_DIS	.EQU M_28H.1	;display flag
F_CAL	.EQU M_28H.2	;calibration flag
F_RP1	.EQU M_28H.3	;RP1 relay flag
F_UOK	.EQU M_28H.4	;U OK flag
F_RP2	.EQU M_28H.5	;RP2 relay flag
F_LDR	.EQU M_28H.6	;LDR relay flag

U_CH	.EQU 029H		;current U channel
U_CH.0	.EQU M_29H.0	;U_CH bit 0
U_CH.1	.EQU M_29H.1	;U_CH bit 1
U_CH.2	.EQU M_29H.2	;U_CH bit 2

FANF	.EQU M_2AH.0	;fan force flag
FANH	.EQU M_2AH.1	;fan hi-force flag
ADCPR	.EQU M_2AH.2	;ADC present flag
T1_PR	.EQU M_2AH.3	;thermometer 1 present flag
T2_PR	.EQU M_2AH.4	;thermometer 2 present flag
T1_PRS	.EQU M_2AH.5	;thermometer 1 on start present flag
T2_PRS	.EQU M_2AH.6	;thermometer 2 on start present flag

WPLC	.EQU 02BH		;WPL copy
WPHC	.EQU 02CH		;WPH copy
WLC	.EQU 02DH		;WL copy
WHC	.EQU 02EH		;WH copy

;  Internal data memory

CODE	.EQU 030H		;control code

DIG1	.EQU 031H		;DIG1 copy
DIG2	.EQU 032H		;DIG2 copy
DIG3	.EQU 033H		;DIG3 copy
DIG4	.EQU 034H		;DIG4 copy

TEMPH	.EQU 035H		;temperature high byte
TEMPL	.EQU 036H		;temperature low  byte

ZCAL	.EQU 037H		;ADC zero calibration cycles counter
ZELL	.EQU 038H		;ADC zero error left channel low
ZELH	.EQU 039H		;ADC zero error left channel high
ZERL	.EQU 03AH		;ADC zero error right channel low
ZERH	.EQU 03BH		;ADC zero error right channel high

WH	.EQU 03CH		;power high byte
WL	.EQU 03DH		;power low  byte
WS	.EQU 03EH		;power sub  byte

WPH	.EQU 03FH		;peak power high byte
WPL	.EQU 040H		;peak power low  byte

SDUR	.EQU 041H		;sound duration
FREQL	.EQU 042H		;sound frequency low
FREQH	.EQU 043H		;sound frequency high

DISPMD	.EQU 044H		;display mode (1-M, 2-G, 3-WP, 4-W, 5-OFF)
DISPRE	.EQU 045H		;display pre-mode
DISPBR	.EQU 046H		;display brightness
DISPWM	.EQU 047H		;display PWM
DISFTM	.EQU 048H		;display fade timer

NOW_MD	.EQU 049H		;now mode code
MIS_MD	.EQU 04AH		;missing mode code

OFFTM	.EQU 04BH		;off timer (1 min)
AUFTM	.EQU 04CH		;auto off timer (1 min)
FANTM	.EQU 04DH		;fan timer (1 S)
UWTTM	.EQU 04EH		;U wait timer (1S)
DELTM	.EQU 04FH		;delay timer (20mS)
KEYHTM	.EQU 050H		;key hold delay timer (20mS)
DISPTM	.EQU 051H		;display timer (20mS)
TEMPTM	.EQU 052H		;thermometer ready timer (20mS)
PHLDTM	.EQU 053H		;peak power hold timer (20mS)
PDECTM	.EQU 054H		;power dec timer (20mS)

PPDM	.EQU 055H		;power display mode

;  External data memory

REG1	.EQU 000H		;hardware register U1 (74AC374)
REG2	.EQU 001H		;hardware register U2 (74AC374)
REG3	.EQU 002H		;hardware register U3 (74AC374)
REG4	.EQU 003H		;hardware register U4 (74AC374)
REG5	.EQU 004H		;hardware register U5 (74AC374)

;  ------ Vectors Area ------

	.ORG 0000H	;reset vector
	LJMP INIT
	.ORG 0013H	;INT1 vector
	LJMP RC5
	.ORG 000BH	;INT TIMER 0 vector
	LJMP RTC
	.ORG 001BH	;INT TIMER 1 vector
	LJMP SND

;  ------ Main Program ------

;Initialization:

INIT:	MOV SP,#STACK	;stack init
	CLR SCLK		;SCLK line init
	CLR ADCPR		;clear ADC present bit
	CLR T1_PR		;clear thermometer 1 present flag
	CLR T2_PR		;clear thermometer 2 present flag
	LCALL INDCLR	;clear hardware registers
	LCALL LEDCLR	;clear non-display LEDs
	LCALL AMUX_G	;ground ADC input
	MOV R37,#NONE
	MOV CODE,R37	;control code clear
	SETB EX1_EN	;set EX1 enable flag
	CLR KEYHE		;clear key hold enable flag
	CLR EXTLVE	;clear EXT leave enable flag
	CLR A
	MOV FAIL_L,A	;left channel fail flags clear
	MOV FAIL_R,A	;right channel fail flags clear
	MOV DISPTM,A	;display timer clear
	MOV KEYHTM,A	;standby hold delay timer clear
	MOV UWTTM,A	;U wait timer clear
	MOV PHLDTM,A	;peak power hold timer clear
	MOV PDECTM,A	;power dec timer clear
	MOV OFFTM,A	;off timer clear
	MOV FANTM,A	;fan timer clear
	MOV STATES,A	;states clear
	LCALL CLR_POW	;WS,WL,WH,WPL,WPH,WLC,WHC,WPLC,WPHC clear
	CLR OFF_EN	;clear off enable flag
	CLR FANF		;clear fan force flag
	CLR FANH		;clear fan hi-force flag
	MOV AUFTM,#AUFTMV	;auto off timer load
	MOV DISPBR,A	;display brightness init
	MOV DISPRE,#DM_W	;display pre-mode init
	MOV PPDM,#DM_W	;save power display mode
	MOV DISPMD,#DM_F	;display mode init
	MOV DISFTM,#DISFTMV	;display fade timer load
	LCALL SETLEDS	;set LEDS to display off mode
	LCALL DISP	;display clear

	LCALL START_T1	;start conversion for thermometer 1
	LCALL START_T2	;start conversion for thermometer 2
	MOV TEMPTM,#TEMPTMV	;thermometer ready timer load

	MOV NOW_MD,#ST_MD	;set standby mode
	MOV MIS_MD,#OP_MD	;set missing operate mode
	MOV DELTM,#INIDEL	;load initial operate on delay

	MOV TMOD,#11H    	;timer 0 and timer 1 init
	CLR TR0          	;timer 0 stop
	CLR TR1          	;timer 1 stop
	MOV TL0,#LO(RTCV)  	;timer 0 load
	MOV TH0,#HI(RTCV)
	MOV RTPC,#0H     	;program counter load
	MOV RTPCS,#V1S   	;1S   value load
	MOV RTPCM,#V1M	;1MIN value load

	SETB PX1          	;high   INT1 priority
	SETB IT1         	;fall   INT1 activating
	SETB EX1		;enable INT1
	CLR PT0          	;timer 0 int low priority
	SETB ET0         	;timer 0 int enable
	SETB TR0         	;timer 0 start
	SETB PT1         	;timer 1 int high priority
	SETB ET1         	;timer 1 int enable
	SETB EA          	;interrupts enable

	MOV A,#01H	;beep frequency
	LCALL SOUND	;initial beep

;Main Loop:

MAIN:     JNB EA,MA11
	MOV P0,#0H	;local keyboard test
	MOV C,RETL
	MOV P0,#0FFH
	JNC MA10		;jump if press
	SETB PRESS
	SJMP MA11		;jump if no press

MA10:	LCALL SCAN
	MOV R2,A		;save keyboard code
	LCALL DEL10
	LCALL SCAN
	CJNE A,R02,MA11	;no press if codes are differ

	CLR EX1		;RC int disable
	ANL R37,#0C0H	;clear com. bits
	ORL A,R37		;set com. bits
	JNB PRESS,MA12
	CPL ACC.6		;if PRESS = 1 then cpl bit 6
	CLR PRESS		;clear PRESS
MA12:    	MOV R37,A		;R37 load, end keyboard holding

MA11:	CLR EX1		;RC int disable
	MOV A,R37		;command code load
	ANL R37,#0C0H	;clear com. bits
	MOV C,EX1_EN
	MOV EX1,C		;EX1_EN copy to EX1
	MOV R0,A
	ANL A,#3FH
	JZ MA20		;no press or unused code
	MOV A,R0
	CJNE A,CODE,DIF	;jump if new command
	ANL A,#3FH	;control bits clear
	MOV R7,A
	SJMP SAME

DIF:	MOV CODE,A	;new press, store code
	ANL A,#3FH	;control bits clear
	MOV R7,A

;Control Functions Analyzis and Processing:

	CJNE R7,#CODSTB,$+9H	;STANDBY
	LCALL SOUND		;beep
	LCALL STANDBY

	MOV AUFTM,#AUFTMV		;auto off timer load
	JNB F_DIS,MA20		;skip cntrl functions if display off
	MOV A,R7
	LCALL SOUND		;beep

	CJNE R7,#CODDIS,$+6H	;DISPLAY
	LCALL DISPLAY
	CJNE R7,#CODMOD,$+6H	;MODE
	LCALL MODE
	CJNE R7,#CODMIN,$+6H	;MINUS
	LCALL MINUS
	CJNE R7,#CODPKP,$+6H	;PEAK
	LCALL PEAK
	CJNE R7,#CODTIM,$+6H	;TIMER
	LCALL TIMER
	CJNE R7,#CODTMP,$+6H	;TEMP
	LCALL TEMP
	CJNE R7,#CODPLS,$+6H	;PLUS
	LCALL PLUS

SAME:	CJNE R7,#CODSTB,$+6H	;STANDBY HOLD
	LCALL STBHLD
	CJNE R7,#CODMOD,$+6H	;MODE HOLD
	LCALL MODHLD
	CJNE R7,#CODMIN,$+6H	;MINUS HOLD
	LCALL MNSHLD
	CJNE R7,#CODPLS,$+6H	;PLUS HOLD
	LCALL PLSHLD
	CJNE R7,#CODTIM,$+6H	;TIMER HOLD
	LCALL TMRHLD
	CJNE R7,#CODDIS,$+6H	;DISPLAY HOLD
	LCALL DISHLD
	CJNE R7,#CODPKP,$+6H	;PEAK HOLD
	LCALL PEAKHLD

;End of Control Functions Analyzis.

;Modes Analyzis and Processing:

MA20:	MOV A,NOW_MD
	XRL A,MIS_MD
	JZ MA30			;skip if NOW_MD = MIS_MD
	MOV R7,MIS_MD
	CJNE R7,#ST_MD,MA21
	LCALL DO_ST		;do STANDBY
MA21:	CJNE R7,#OP_MD,MA22
	LCALL DO_OP		;do OPERATE
MA22:	CJNE R7,#FL_MD,MA30
	LCALL DO_FL		;do FAIL

;End of Modes Analyzis and Processing.

;Temperature measure:

MA30:     JNB F_DIS,MA40	;skip block if display off
	MOV A,TEMPTM	;thermometer ready timer check
	JNZ MA40
	MOV TEMPTM,#TEMPTMV	;thermometer ready timer reload
	LCALL READ_T	;read T, update TEMPH, TEMPL and control TF
	LCALL START_T1	;start conversion for thermometer 1
	LCALL START_T2	;start conversion for thermometer 2

;Voltage measure:

MA40:     JNB F_RP1,MA50	;skip block if relay 1 off
	LCALL SET_UCH	;set U channel
	LCALL ADC_UCH	;measure U1-U4 -> UF, UR (in FAIL_L,R)

;Protection check:

MA50:     JNB F_UOK,MA60	;skip block if U not OK
	LCALL PR_CH	;OKL,OKR,DCF,IOF -> IF,DF,PF (in FAIL_L,R)

;Power measure:

MA60:     JNB F_LDR,MA70		;skip block if load off
	LCALL GET_POW		;meas. power and upd. WH,WL,WPH,WPL
	JNB ADCPR,MA62
	MOV A,#(PALIM>>7) & 255	;load HI(PALIM*2)
	CJNE A,WH,MA61		;check for limit
	MOV A,#(PALIM<<1) &	255	;load LO(PALIM*2)
	CJNE A,WL,MA61
MA61:	JNC MA70			;jump if WH,WL<=PALIM
MA62:	MOV AUFTM,#AUFTMV		;auto off timer load

;Fan control:

MA70:     JNB F_DIS,MA80		;skip block if display off
	JB FANH,MA74		;fan on if FANH=1
	JNB ADCPR,MA72
	MOV A,#(PNLIM>>7) & 255	;load HI(PNLIM*2)
	CJNE A,WH,MA71		;check for limit
	MOV A,#(PNLIM<<1) &	255	;load LO(PNLIM*2)
	CJNE A,WL,MA71
MA71:	JC MA72			;if WH,WL>PNLIM go to check timer

MA73:	MOV FANTM,#FANTMV		;fan timer reload
	SETB FAN			;fan off
	SJMP MA80

MA72:	MOV A,FANTM		;fan timer check
	JNZ MA80			;jump if no timer overflow
	JNB FANF,MA73		;fan off if FANF=0
MA74:	CLR FAN			;fan on

;Auto off and off timers:

MA80:	JNB F_DIS,MA90	;skip block if display off
	MOV A,OFFTM	;check OFFTM for 0
	JNZ NTO		;skip if no timer overflow
	JBC OFF_EN,DOF	;OFFTM = 0, do off if off enable

	MOV A,AUFTM	;check AUFTM for 1
	CJNE A,#1H,MA90	;skip if AUFTM <> 1
	MOV AUFTM,#AUFTMV	;auto off timer load
#IFDEF AUTO_OFF
	MOV OFFTM,A	;OFFTM load
	SETB LEDTMR	;LED TIMER on
#ENDIF
	SJMP MA90

NTO:	JB OFF_EN,MA90	;skip if OFF_EN = 1
	DEC A		;check OFFTM for 1
	JNZ MA90		;skip if OFFTM > 1
	SETB OFF_EN	;set off enable flag
	LCALL BEEP	;off timer beep
	LCALL TIMER	;set timer indication mode
	SJMP MA90

DOF:	CLR LEDTMR	;clear timer LED
	MOV MIS_MD,#ST_MD	;load STANDBY missing mode

;Fail control:

MA90:	JNB F_UOK,MA100	;skip block if U not OK
	MOV A,FAIL_L
	ORL A,FAIL_R
	JZ MA100		;jump if FAIL_L,R=0
	MOV MIS_MD,#FL_MD	;set missing function FAIL

;Display:

MA100:	MOV A,DISPBR	;display brightness check
	JNZ BR_CH		;brightness change if DISPBR > 0

	MOV A,DISPTM	;display timer check
	JNZ MA200
	MOV DISPTM,#DISPTMV	;display timer reload
	LCALL COPY	;copy W, WP
	LCALL DISP	;display W or WP or G or M
	SJMP MA200

BR_CH:    JB TR1,MA200	;skip display fade if sound

	CLR EA		;brightness change, interrupts disable
	CPL WD

	CLR C
	SUBB A,DISPWM	;DISPBR - DISPWM
	JC DDFC1
	MOV A,DISPRE	;exchange DISPRE and DISPMD
	XCH A,DISPMD
	MOV DISPRE,A
	LCALL SETLEDS	;set LEDS with DISPRE
	LCALL DISP	;display DISPRE
	MOV A,DISPRE	;restore DISPRE and DISPMD
	XCH A,DISPMD
	MOV DISPRE,A
	SJMP PWM
DDFC1:	LCALL SETLEDS	;set LEDS with DISPMD
	LCALL DISP	;display DISPMD

PWM:	DEC DISPWM	;next PWM phase
	ANL DISPWM,#0FH

	DEC DISFTM	;dec display fade timer
	MOV A,DISFTM	;display fade timer check
	JNZ MA200		;jump if no overflow
	MOV DISFTM,#DISFTMV	;display fade timer reload

	MOV A,DISPBR	;next brightness step
	MOV B,#SLEW
	DIV AB
	INC A
	XCH A,DISPBR
	CLR C
	SUBB A,DISPBR
	JNC BR_LD
	CLR A
BR_LD:	MOV DISPBR,A

	JNZ MA200		;brightness check, jump if DISPBR <> 0
	LCALL SETLEDS	;set LEDS with DISPMD
	LCALL DISP	;display DISPMD
	SETB EA		;end of display fade, interrupts enable

;Watchdog Wakeup:

MA200:	MOV C,T040M
	MOV WD,C

	LJMP MAIN

;  ------ Subroutines Area ------

;  ------ Control Functions Processing ------

;Key "STANDBY" processing

STANDBY:  MOV KEYHTM,#STBHTMV	;key hold timer load
	SETB KEYHE	;set key hold enable flag
	MOV A,NOW_MD	;check now mode
	XRL A,#PR_MD
	MOV DELTM,#STRDEL	;delay timer load if processing
	JZ STB2
	MOV DELTM,#0H	;clear delay timer
STB2:	MOV A,MIS_MD	;missing mode check
	XRL A,#ST_MD
	JNZ STB3
	MOV MIS_MD,#OP_MD	;set OPERATE missing mode...
	SJMP STB1
STB3:	MOV MIS_MD,#ST_MD	;or standby missing mode
	CLR EXTLVE	;clear EXT leave enable flag
STB1:	RET

STBHLD:   MOV A,KEYHTM	;key hold timer check
	JNZ SH1		;skip if no overflow
	JNB KEYHE,SH1	;skip if key hold enable flag = 0
	CLR KEYHE		;clear key hold enable flag
	LCALL HLDSND	;key hold sound
	CPL EXTLVE	;set/clear EXT leave enable flag
	MOV DELTM,#0H	;clear delay timer
	MOV MIS_MD,#ST_MD	;set STANDBY missing mode
SH1:	RET

;Key "DISPLAY" processing

DISPLAY:  MOV KEYHTM,#STBHTMV	;key hold timer load
	SETB KEYHE	;set key hold enable flag
	MOV A,DISPMD	;load current display mode
	MOV DISPRE,A	;set display pre-mode = display mode
	INC A		;next display mode
	CJNE A,#DM_F+1,NDMDO
	MOV A,#DM_W
NDMDO:	LCALL DSPMDA	;set new display mode
	RET

DISHLD:	MOV A,KEYHTM	;key hold timer check
	JNZ DH1		;skip if no overflow
	JNB KEYHE,DH1	;skip if key hold enable flag = 0
	CLR KEYHE		;clear key hold enable flag
	MOV A,DISPMD	;load current display mode
	CJNE A,#DM_F,DH2
	SJMP DH1		;skip if display off
DH2:	MOV DISPRE,A	;set display pre-mode = display mode
	LCALL HLDSND	;key hold sound
	MOV A,#DM_F	;display off
	LCALL DSPMDA	;set new display mode
DH1:	RET

;Key "PEAK" processing

PEAK:     MOV KEYHTM,#STBHTMV	;key hold timer load
	SETB KEYHE	;set key hold enable flag
	MOV A,DISPMD
	MOV DISPRE,A	;set display pre-mode = display mode
	CJNE A,#DM_WP,PK1	;display mode check
	MOV A,#DM_W	;set power display mode if DISPMD = DM_WP...
	MOV PPDM,A	;save power display mode
	LCALL DSPMDA	;set new display mode
	RET
PK1:	CJNE A,#DM_W,PK2	;display mode check
	MOV A,#DM_WP	;set peak power display mode if DISPMD = DM_W...
	MOV PPDM,A	;save power display mode
	LCALL DSPMDA	;set new display mode
	RET
PK2:	MOV A,PPDM	;else set previous power display mode
	LCALL DSPMDA	;set new display mode
	RET

PEAKHLD:	MOV A,KEYHTM	;key hold timer check
	JNZ KH1		;skip if no overflow
	JNB KEYHE,KH1	;skip if key hold enable flag = 0
	CLR KEYHE		;clear key hold enable flag
	MOV A,DISPMD	;load current display mode
	CJNE A,#DM_F,KH2
	SJMP KH1		;skip if display off
KH2:	MOV DISPRE,A	;set display pre-mode = display mode
	LCALL HLDSND	;key hold sound
	MOV A,#DM_F	;display off
	LCALL DSPMDA	;set new display mode
KH1:	RET

;Key "TIMER" processing

TIMER:	MOV KEYHTM,#TMRHTMV	;key hold timer load
	SETB KEYHE	;set key hold enable flag
	MOV A,DISPMD
	XRL A,#DM_M	;display mode check
	JZ TMRR
	MOV DISPRE,DISPMD	;set display pre-mode = display mode
	MOV A,#DM_M	;timer display mode
	LCALL DSPMDA	;set new display mode
TMRR:	RET

TMRHLD:	MOV A,KEYHTM
	JNZ TMH1
	JNB KEYHE,TMH1	;skip if key hold enable flag = 0
	CLR KEYHE		;clear key hold enable flag
	MOV A,OFFTM
	JZ TMH1		;skip if OFFTM = 0
	LCALL HLDSND	;key hold sound
	MOV OFFTM,#0H	;clear off timer
	CLR LEDTMR	;if OFFTM = 0 then timer led off
	CLR OFF_EN	;clear off enable flag
	MOV DISPTM,#0H	;display timer clear
TMH1:	RET

;Key "TEMP" processing

TEMP:	MOV A,DISPMD
	XRL A,#DM_G	;display mode check
	JZ TMPR
	MOV DISPRE,DISPMD	;set display pre-mode = display mode
	MOV A,#DM_G	;temperature display mode
	LCALL DSPMDA	;set new display mode
TMPR:	RET

;Key "MODE" processing

MODE:     MOV KEYHTM,#KEYID	;load key hold initial delay
	SJMP MD6
MODHLD:	MOV A,KEYHTM	;key repeat timer check
	JNZ MD1
	MOV KEYHTM,#KEYDD	;load key hold autorepeat delay
MD6:	MOV A,DISPMD	;load current display mode
	CJNE A,#DM_M,TIMER	;set timer display mode
	MOV RTPCM,#V1M	;1MIN program counter reload
	JNB OFF_EN,MD5
	MOV R3,#0H	;if OFF_EN = 1 then clear timer
	SJMP MD3
MD5:	MOV R0,OFFTM	;load current off timer value
	MOV R1,#0H	;pointer init

MD2:	MOV A,R1		;load pointer
	INC R1		;INC pointer
	MOV DPTR,#OFFTBL	;load table base address
	MOVC A,@A+DPTR	;read table
	MOV R3,A		;save new value
	JZ MD3		;if new value = 0 then OFFTM = 0
	CLR C
	MOV A,R0
	SUBB A,R3
	JNC MD2		;loop if new value <= OFFTM

MD3:      MOV A,R3
	MOV OFFTM,A	;save new value in OFFTM
	SETB LEDTMR	;timer LED on
	JNZ MD4
	CLR LEDTMR	;if OFFTM = 0 then timer LED off
	CLR OFF_EN	;clear off enable flag
MD4:	MOV DISPTM,#0H	;display timer clear

MD1:	RET

;Key "PLUS" processing

PLUS:     MOV KEYHTM,#KEYID	;load key hold initial delay
	SETB KEYHE	;set key hold enable flag
	SJMP PL4
PLSHLD:	MOV A,KEYHTM	;key repeat timer check
	JNZ PL1
	JNB KEYHE,PL1
	MOV KEYHTM,#KEYDD	;load key hold autorepeat delay
PL4:	MOV A,DISPMD	;load current display mode
	CJNE A,#DM_M,PL1	;return if display mode <> DM_M
	MOV RTPCM,#V1M	;1MIN program counter reload
	MOV R0,OFFTM	;load current off timer value
	MOV R1,#0H	;pointer init

PL2:	MOV A,R1		;load pointer
	INC R1		;INC pointer
	MOV DPTR,#OFFTBL	;load table base address
	MOVC A,@A+DPTR	;read table
	JZ PL3		;if new value = 0 then stop table scan
	MOV R3,A		;save new value
	CLR C
	MOV A,R0
	SUBB A,R3
	JNC PL2		;loop if new value <= OFFTM

PL3:      JNZ PL5
	LCALL BEEP	;limit beep
	CLR KEYHE
PL5:	MOV OFFTM,R3	;save new value in OFFTM
	SETB LEDTMR	;timer LED on
	MOV DISPTM,#0H	;display timer clear

PL1:	RET

;Key "MINUS" processing

MINUS:	MOV KEYHTM,#KEYID	;load key hold initial delay
	SETB KEYHE	;set key hold enable flag
	SJMP MN4
MNSHLD:   MOV A,KEYHTM	;key repeat timer check
	JNZ MN1
	JNB KEYHE,MN1
	MOV KEYHTM,#KEYDD	;load key hold autorepeat delay
MN4:	MOV A,DISPMD	;load current display mode
	CJNE A,#DM_M,MN1	;return if display mode <> DM_M
	MOV RTPCM,#V1M	;1MIN program counter reload
	MOV R0,OFFTM	;load current off timer value
	MOV R1,#0H	;pointer init
	MOV R2,#0H	;min value init

MN2:	MOV A,R1		;load pointer
	INC R1		;INC pointer
	MOV R3,R02
	MOV DPTR,#OFFTBL	;load table base address
	MOVC A,@A+DPTR	;read table
	MOV R2,A		;save new value
	CLR C
	SUBB A,R0
	JC MN2		;loop if new value < OFFTM

	MOV A,R3
	JNZ MN3
	MOV A,OFFTM
	JNZ MN5
	LCALL BEEP	;limit beep
	CLR KEYHE
MN5:	CLR LEDTMR	;if OFFTM = 0 then timer led off
	CLR OFF_EN	;clear off enable flag
	MOV A,R3
MN3:	MOV OFFTM,A	;save new value in OFFTM
	MOV DISPTM,#0H	;display timer clear

MN1:	RET

;  ------ End of Control Functions Processing ------

;Display mode set
;Input: A = new display mode

DSPMDA:	MOV DISPMD,A	;save next display mode
	MOV DISPTM,#0H	;display timer clear
#IFDEF DISP_FADE
	MOV DISPBR,#0FH
#ELSE
	LCALL SETLEDS	;set LEDs according DISPMD
#ENDIF
	RET

;  Display modes codes

DM_W	.EQU 001H		;power indication display mode
DM_WP	.EQU 002H		;peak power indication display mode
DM_M	.EQU 003H		;timer indication display mode
DM_G	.EQU 004H		;temperature indication display mode
DM_F	.EQU 005H		;off display

DM_INI	.EQU DM_F		;initial display mode

;  ------ Modes Processing ------

; Modes Codes

ST_MD	.EQU 001H	;standby code
OP_MD	.EQU 002H ;operate code
FL_MD	.EQU 003H	;fail code
PR_MD	.EQU 000H	;processing in progress

;  Operate function processing

DO_OP:    MOV NOW_MD,#PR_MD	;set processing code
	MOV A,DELTM	;delay timer check
	JZ OP1
	RET

OP1:      JB F_EXT,OP2	;-------------> step 1:
	CLR REX		;EXT relay on
	CLR PEN		;power enable
	SETB LEDEXT	;LED EXT on
	LCALL IND
	MOV DELTM,#EXPDEL
	SETB F_EXT	;step 1 done
	RET

OP2:      JB F_DIS,OP3	;-------------> step 2:
#IFNDEF RESTORE_DM
	MOV DISPRE,#DM_INI
#ENDIF
	MOV A,#DM_F	;set DISPRE to DM_F and
	XCH A,DISPRE	;restore DISPMD from DISPRE
	LCALL DSPMDA	;set LEDs and DISPMD
	MOV DELTM,#DYSDEL
	SETB F_DIS	;step 2 done
	RET

OP3:      JB F_CAL,OP4	;-------------> step 3:
	SETB START
	SETB SDATA
	LCALL ADC_DET	;detect ADC present
	CLR A		;perform calibration process:
	MOV ZELH,A	;ZELH clear
	MOV ZELL,A	;ZELL clear
	MOV ZERH,A	;ZERH clear
	MOV ZERL,A	;ZERL clear
	MOV ZCAL,#16	;load zero calibration cycles number
	LCALL IND
	SETB F_CAL	;step 3 done
	RET

OP4:      MOV A,ZCAL	;-------------> step 4:
	JZ OP5		;skip block if ZCAL=0 (calibration done)
	LCALL ZCALIB	;do zero calibration, DEC ZCAL
	RET

OP5:      JB F_RP1,OP6	;-------------> step 5:
	SETB LEDMNP	;LED MAIN on
	CLR RP1		;POWER 1 relay on
	SETB LEDMNP	;LED MAIN on
	LCALL IND
	MOV UWTTM,#UWTDEL	;load U wait timer
	MOV DELTM,#RP1DEL
	SETB F_RP1	;step 5 done
	RET

OP6:      JB F_UOK,OP7	;-------------> step 6:
	MOV C,UF_L
	ORL C,UF_R
	JNC U_OKE		;if UF_L and UF_R = 0 then U is OK
	MOV A,UWTTM	;U wait timer check
	JNZ NO_TOVF	;return if no timer overflow
	MOV MIS_MD,#FL_MD	;set missing function FAIL
	SJMP NO_TOVF
U_OKE:    MOV DELTM,#RP2DEL
	SETB F_UOK	;step 6 done
NO_TOVF:	RET

OP7:      JB F_RP2,OP8	;-------------> step 7:
	CLR RP2		;POWER 2 relay on
	SETB LEDOPR	;LED OPERATE on
	LCALL IND
	MOV DELTM,#LDRDEL
	SETB F_RP2	;step 7 done
	RET

OP8:	MOV A,FAIL_L	;-------------> step 8:
	ORL A,FAIL_R
	JNZ F_LR
	SETB LEDLDR	;all OK, LED LOAD on
	CLR ENB		;enable load relay
	LCALL IND
	SETB F_LDR	;step 8 done
	MOV NOW_MD,#OP_MD	;end of function
F_LR:	RET

;  Standby function processing

DO_ST:    MOV NOW_MD,#PR_MD	;set processing code
	MOV A,DELTM	;delay timer check
	JZ ST1
	RET

ST1:      JNB F_LDR,ST2	;<------------- step 8:
	CLR LEDLDR	;LED LOAD off
	SETB ENB		;disable load relay
	MOV OFFTM,#0H	;off timer clear
	CLR LEDTMR	;LED TIMER off
	LCALL IND
	LCALL CLR_POW	;WS,WL,WH,WPL,WPH,WLC,WHC,WPLC,WPHC clear
	MOV DELTM,#LDFDEL
	CLR F_LDR		;step 8 undone
	RET

ST2:      JNB F_RP2,ST3	;<------------- step 7:
	CLR LEDOPR	;LED OPERATE off
	SETB RP2		;POWER 2 relay off
	LCALL IND
	MOV DELTM,#P2FDEL	;load delay value
	CLR F_RP2		;step 7 undone
	RET

ST3:      JNB F_UOK,ST4	;<------------- step 6:
	CLR F_UOK		;step 6 undone
	RET

ST4:      JNB F_RP1,ST5	;<------------- step 5:
	CLR LEDMNP	;LED MAIN off
	SETB RP1		;POWER 1 relay off
	LCALL IND
	MOV DELTM,#P1FDEL	;load delay value
	CLR F_RP1		;step 5 undone
	RET

ST5:      JNB F_CAL,ST6	;<------------- step 4,3:
	CLR F_CAL		;step 4,3 undone
	RET

ST6:      JNB F_DIS,ST7	;<------------- step 2:
	MOV A,DISPMD
	MOV DISPRE,A	;display mode store
	MOV A,#DM_F	;display off mode
	LCALL DSPMDA	;set LEDs and DISPMD
	MOV FAIL_L,#0H	;FAIL_L clear
	MOV FAIL_R,#0H	;FAIL_R clear
	CLR FANF		;clear fan force flag
	CLR FANH		;clear fan hi-force flag
	SETB FAN		;fan off
	LCALL IND
	MOV DELTM,#EXFDEL
	CLR F_DIS		;step 2 undone
	RET

ST7:      JB EXTLVE,ST71	;<------------- step 1:
	SETB REX		;EXT relay off
	LCALL AMUX_G	;ground ADC input
	CLR START
	CLR SDATA
	SETB PEN		;power disable
	CLR LEDEXT	;EXTLVE = 0, LED EXT off
	LCALL IND		;indicate
	CLR F_EXT		;step 1 undone
ST71:	MOV NOW_MD,#ST_MD	;end of function
	RET

;  Fail function processing

DO_FL:    SETB EA
	CLR LEDLDR	;LED LOAD off
	SETB ENB		;disable load relay
	MOV OFFTM,#0H	;off timer clear
	CLR LEDTMR	;LED TIMER off
	CLR OFF_EN	;clear off enable flag
	LCALL IND
	CLR F_LDR		;step 8 undone

	LCALL FAILDEL	;delay

	CLR LEDOPR	;LED OPERATE off
	SETB RP2		;POWER 2 relay off
	LCALL IND		;indicate
	CLR F_RP2		;step 7 undone

	CLR F_UOK		;step 6 undone

	LCALL FAILDEL	;delay

	CLR LEDMNP	;LED MAIN off
	SETB RP1		;POWER 1 relay off
	CLR F_RP1		;step 5 undone

	CLR F_CAL		;step 4,3 undone

	MOV DISPRE,DISPMD	;display mode store
	MOV DISPMD,#DM_W	;power display mode (fail information)
	MOV DISPTM,#0H	;display timer clear
	LCALL SETLEDS	;set LEDs according DISPMD

	MOV NOW_MD,#FL_MD	;end of function
	RET

;  ------ End of Modes Processing ------

;Fail process delay (#FLDEL x 20 mS)

FAILDEL:	MOV DELTM,#FLDEL	;load delay value
FLDL1:	MOV A,DELTM	;delay
	MOV C,T040M	;watchdog wakeup
	MOV WD,C
	JNZ FLDL1
	RET

;Clear WS,WL,WH,WPL,WPH,WLC,WHC,WPLC,WPHC

	LCALL CLR_POW	;WS,WL,WH,WPL,WPH,WLC,WHC,WPLC,WPHC clear
CLR_POW:  CLR A
	MOV WS,A		;WS clear
	MOV WL,A		;WL clear
	MOV WH,A		;WH clear
	MOV WPL,A		;WPL clear
	MOV WPH,A		;WPH clear
	MOV WLC,A		;WL copy clear
	MOV WHC,A		;WH copy clear
	MOV WPLC,A	;WPL copy clear
	MOV WPHC,A	;WPH copy clear
	RET

;  Set LEDs according DISPMD
;  Input: DISPMD
;  Out: POINT,LEDWTT,LEDPKP,LEDDGR,LEDTME

SETLEDS:	MOV A,DISPMD
	CJNE A,#DM_W,DSP2	;power indication mode
	SETB POINT
	SETB LEDWTT
	CLR LEDPKP
	CLR LEDDGR
	CLR LEDTME
	RET
DSP2:	CJNE A,#DM_WP,DSP3	;peak power indication mode
	SETB POINT
	SETB LEDWTT
	SETB LEDPKP
	CLR LEDDGR
	CLR LEDTME
	RET
DSP3:	CJNE A,#DM_G,DSP4	;temperature indication mode
	SETB POINT
	CLR LEDWTT
	CLR LEDPKP
	SETB LEDDGR
	CLR LEDTME
	RET
DSP4:	CJNE A,#DM_M,DSP5	;timer indication mode
	CLR POINT
	CLR LEDWTT
	CLR LEDPKP
	CLR LEDDGR
	SETB LEDTME
	RET
DSP5:	CLR POINT
	CLR LEDWTT
	CLR LEDPKP
	CLR LEDDGR
	CLR LEDTME
	RET

;Local Keyboard Scan

SCAN:    	MOV R1,#0H	;scan-code init
	MOV A,#0FEH	;shifting byte init
	SETB C
SCAN2:    MOV P0,A
	JNB RETL,SCAN1	;jump if pressed key found
	INC R1
	RLC A
	JC SCAN2
SCAN1:    MOV P0,#0FFH
	MOV A,R1
	MOV DPTR,#KEYTAB
	MOVC A,@A+DPTR
	RET

;  Beeper support

;Key hold sound

HLDSND:	MOV A,#9H		;load sound frequency
	LCALL SOUND	;beep generation
	RET

;Sound with frequency from A
;A=XXXX0000B - 977  Hz
;A=XXXX1111B - 1838 Hz

SOUND:    CLR TR1		;timer 1 stop
	ADD A,#7H
	SWAP A
	MOV FREQL,A	;frequency low byte save
	MOV FREQH,#0FEH	;frequency high byte save
	CLR C
	RRC A
	ADD A,#070H	;sound duration correction
	MOV SDUR,A	;sound duration load
	MOV TL1,FREQL	;timer 1 load
	MOV TH1,FREQH
	SETB TR1		;timer 1 start
	RET

;Beep generation

BEEP:	CLR EX1		;keyboard interrupt disable
	CLR EX1_EN	;keyboard interrupt enable flag clear
	CLR TR1		;timer 1 stop
	MOV FREQL,#0E0H	;frequency low byte save
	MOV FREQH,#0FDH	;frequency high byte save
	MOV SDUR,#0FFH      ;sound duration load
	MOV TL1,FREQL	;timer 1 load
	MOV TH1,FREQH
	SETB TR1		;timer 1 start
	RET

;Delay 10mS

DEL10:    PUSH R02         ;delay 10mS
	PUSH R03
	MOV R2,#20
	LCALL DEL05
	POP R03
	POP R02
	RET

DEL05:	MOV R3,#250      ;delay R2x0.5mS, uses R2,R3 (for 12MHz)
	DJNZ R3,$
	MOV C,T040M
	MOV WD,C	       ;watchdog wakeup
	DJNZ R2,DEL05
	RET

;  ------ 1-Wire Dallas Interface Support Libraries ------
;
;  Uses CLK_8MHZ, CLK_10MHZ, CLK_11MHZ or CLK_12MHZ
;
;  Main program must define:
;
;  Dallas 1-Wire Port as OWP1,2
;
;  This library includes:
;
;  TRESET1,2  - Reset pulse generation for port #1
;  TOUCHBY1,2 - Read/write byte via port #1
;
;  TRESET1,2:
;  This procedure transmits the Reset signal to the 1-Wire
;  device and watches for a presence pulse. On return,
;  the Carry bit is set if a presence pulse was detected,
;  otherwise the Carry is cleared. The code is timed for
;  an 8, 10, 11.059 and 12 MHz crystal.
;
;  TOUCHBY1,2:
;  This procedure sends the byte in the accumulator
;  to the 1-Wire device, and returns a byte from the
;  1-Wire device in the accumulator. For reading
;  operation set accumulator to 0FFH. The code is
;  timed for an 8, 10, 11.059 and 12 MHz crystal.

#DEFINE CLK_12MHZ		;OSC frequency

;  ------ TRESET1 ------

TRESET1:	CLR  OWP1      		;Start the reset pulse.
#IFDEF CLK_8MHZ  \ MOV B,#160 \ #ENDIF	;  2.  Set time interval.
#IFDEF CLK_10MHZ \ MOV B,#200 \ #ENDIF	;  2.  Set time interval.
#IFDEF CLK_11MHZ \ MOV B,#221 \ #ENDIF	;  2.  Set time interval.
#IFDEF CLK_12MHZ \ MOV B,#240 \ #ENDIF	;  2.  Set time interval.
	DJNZ B,$         		;320. 400. 442. 480. Wait with Data low.
#IFDEF INT_DIS \ CLR EA \ #ENDIF	; Interrupt disable if needed
	SETB OWP1       		;  1.  Release Data line.
#IFDEF CLK_8MHZ  \ MOV B,#80  \ #ENDIF	;High wait delay.
#IFDEF CLK_10MHZ \ MOV B,#101 \ #ENDIF	;High wait delay.
#IFDEF CLK_11MHZ \ MOV B,#111 \ #ENDIF	;High wait delay.
#IFDEF CLK_12MHZ \ MOV B,#120 \ #ENDIF	;High wait delay.
	CLR  C           		;  1.  Clear presence flag.
WAITL1:	JB OWP1,WH1    		;Exit loop if line high.
	DJNZ B,WAITL1   		;Hang around if line is low.
#IFDEF INT_DIS \ SETB EA \ #ENDIF	; Interrupt enable if needed
	SJMP SHORT1       		;Line could not go high.
WH1:
#IFDEF INT_DIS \ SETB EA \ #ENDIF	; Interrupt enable if needed
#IFDEF CLK_8MHZ  \ MOV B,#80  \ #ENDIF	;Delay for presence detect.
#IFDEF CLK_10MHZ \ MOV B,#101 \ #ENDIF	;Delay for presence detect.
#IFDEF CLK_11MHZ \ MOV B,#111 \ #ENDIF	;Delay for presence detect.
#IFDEF CLK_12MHZ \ MOV B,#120 \ #ENDIF	;Delay for presence detect.
HL1:	ORL  C,/OWP1    		;160. 202. 222. 240. Catch presence pulse.
	DJNZ B,HL1        		;160. 202. 222. 240. Wait with Data high.
SHORT1:	RET              		;Return.

;  ------ TOUCHBY1 ------

TOUCHBY1:	PUSH B           		;Save the B register.
	MOV  B,#8        		;Setup for 8 bits.
B_LOOP1:	RRC  A           		;1. Get bit in carry.
	LCALL TOUCHB1   		;2. Send bit.
	DJNZ B,B_LOOP1  		;2. Get next bit.
	RRC  A           		;   Get final bit in ACC.
	POP  B           		;Restore B register.
	RET              		;Return to caller.

TOUCHB1:
#IFDEF INT_DIS \ CLR EA \ #ENDIF	; Interrupt disable if needed
	CLR OWP1       		; 1.  Start the time slot.
	NOP              		; 1.  Delay to make sure
	NOP              		; 1.    that the Touch Memory
	NOP              		; 1.      sees a low for at
	NOP              		; 1.        least 1 microsecond.
	MOV OWP1,C    		; 2.  Send out the data bit.
	NOP              		; 1.  Delay to give the
	NOP              		; 1.    data returned from
	NOP              		; 1.      the Touch Memory
	NOP              		; 1.        time to settle
	NOP              		; 1.          before reading
	NOP              		; 1.            the bit.
	MOV  C,OWP1     		; 1.  Sample input data bit.
#IFDEF INT_DIS \ SETB EA \ #ENDIF	; Interrupt enable if needed
	PUSH B           		; 2.  Save B register.
#IFDEF CLK_8MHZ  \ MOV B,#13 \ #ENDIF	; 2.  Delay until the end
#IFDEF CLK_10MHZ \ MOV B,#16 \ #ENDIF	; 2.  Delay until the end
#IFDEF CLK_11MHZ \ MOV B,#18 \ #ENDIF	; 2.  Delay until the end
#IFDEF CLK_12MHZ \ MOV B,#20 \ #ENDIF	; 2.  Delay until the end
	DJNZ B,$         		;26. 32. 36. 40. of the time slot.
	POP  B           		;Restore B register.
	SETB OWP1       		;Terminate time slot.
	RET              		;Return to caller.

;  ------ TRESET2 ------

TRESET2:	CLR  OWP2      		;Start the reset pulse.
#IFDEF CLK_8MHZ  \ MOV B,#160 \ #ENDIF	;  2.  Set time interval.
#IFDEF CLK_10MHZ \ MOV B,#200 \ #ENDIF	;  2.  Set time interval.
#IFDEF CLK_11MHZ \ MOV B,#221 \ #ENDIF	;  2.  Set time interval.
#IFDEF CLK_12MHZ \ MOV B,#240 \ #ENDIF	;  2.  Set time interval.
	DJNZ B,$         		;320. 400. 442. 480. Wait with Data low.
#IFDEF INT_DIS \ CLR EA \ #ENDIF	; Interrupt disable if needed
	SETB OWP2       		;  1.  Release Data line.
#IFDEF CLK_8MHZ  \ MOV B,#80  \ #ENDIF	;High wait delay.
#IFDEF CLK_10MHZ \ MOV B,#101 \ #ENDIF	;High wait delay.
#IFDEF CLK_11MHZ \ MOV B,#111 \ #ENDIF	;High wait delay.
#IFDEF CLK_12MHZ \ MOV B,#120 \ #ENDIF	;High wait delay.
	CLR  C           		;  1.  Clear presence flag.
WAITL2:	JB OWP2,WH2    		;Exit loop if line high.
	DJNZ B,WAITL2   		;Hang around if line is low.
#IFDEF INT_DIS \ SETB EA \ #ENDIF	; Interrupt enable if needed
	SJMP SHORT2       		;Line could not go high.
WH2:
#IFDEF INT_DIS \ SETB EA \ #ENDIF	; Interrupt enable if needed
#IFDEF CLK_8MHZ  \ MOV B,#80  \ #ENDIF	;Delay for presence detect.
#IFDEF CLK_10MHZ \ MOV B,#101 \ #ENDIF	;Delay for presence detect.
#IFDEF CLK_11MHZ \ MOV B,#111 \ #ENDIF	;Delay for presence detect.
#IFDEF CLK_12MHZ \ MOV B,#120 \ #ENDIF	;Delay for presence detect.
HL2:	ORL  C,/OWP2    		;160. 202. 222. 240. Catch presence pulse.
	DJNZ B,HL2        		;160. 202. 222. 240. Wait with Data high.
SHORT2:	RET              		;Return.

;  ------ TOUCHBY2 ------

TOUCHBY2:	PUSH B           		;Save the B register.
	MOV  B,#8        		;Setup for 8 bits.
B_LOOP2:	RRC  A           		;1. Get bit in carry.
	LCALL TOUCHB2   		;2. Send bit.
	DJNZ B,B_LOOP2  		;2. Get next bit.
	RRC  A           		;   Get final bit in ACC.
	POP  B           		;Restore B register.
	RET              		;Return to caller.

TOUCHB2:
#IFDEF INT_DIS \ CLR EA \ #ENDIF	; Interrupt disable if needed
	CLR OWP2       		; 1.  Start the time slot.
	NOP              		; 1.  Delay to make sure
	NOP              		; 1.    that the Touch Memory
	NOP              		; 1.      sees a low for at
	NOP              		; 1.        least 1 microsecond.
	MOV OWP2,C    		; 2.  Send out the data bit.
	NOP              		; 1.  Delay to give the
	NOP              		; 1.    data returned from
	NOP              		; 1.      the Touch Memory
	NOP              		; 1.        time to settle
	NOP              		; 1.          before reading
	NOP              		; 1.            the bit.
	MOV  C,OWP2     		; 1.  Sample input data bit.
#IFDEF INT_DIS \ SETB EA \ #ENDIF	; Interrupt enable if needed
	PUSH B           		; 2.  Save B register.
#IFDEF CLK_8MHZ  \ MOV B,#13 \ #ENDIF	; 2.  Delay until the end
#IFDEF CLK_10MHZ \ MOV B,#16 \ #ENDIF	; 2.  Delay until the end
#IFDEF CLK_11MHZ \ MOV B,#18 \ #ENDIF	; 2.  Delay until the end
#IFDEF CLK_12MHZ \ MOV B,#20 \ #ENDIF	; 2.  Delay until the end
	DJNZ B,$         		;26. 32. 36. 40. of the time slot.
	POP  B           		;Restore B register.
	SETB OWP2       		;Terminate time slot.
	RET              		;Return to caller.

;  Start conversion in 1-wire thermometer DS1820.
;  1-wire port line called as OWP1.
;  Calls TRESET1, TOUCHBY1

START_T1: LCALL TRESET1	;Issue reset pulse.
	MOV T1_PRS,C	;Make present bit
	JNC ACSRET1         ;Fail if no part on bus.
	MOV A,#0CCH	;Skip ROM command.
	LCALL TOUCHBY1	;Send command byte.
	MOV A,#044H	;Convert T1 command.
	LCALL TOUCHBY1	;Send command byte.
	SETB C
ACSRET1:	RET

;  Start conversion in 1-wire thermometer DS1820.
;  1-wire port line called as OWP2.
;  Calls TRESET2, TOUCHBY2

START_T2: LCALL TRESET2	;Issue reset pulse.
	MOV T2_PRS,C	;Make present bit
	JNC ACSRET2         ;Fail if no part on bus.
	MOV A,#0CCH	;Skip ROM command.
	LCALL TOUCHBY2	;Send command byte.
	MOV A,#044H	;Convert T2 command.
	LCALL TOUCHBY2	;Send command byte.
	SETB C
ACSRET2:	RET

;  Read temperature from 1-wire thermometer DS1820.
;  Returns C=1 in case of valid code, else C=0.
;  Out   R2,R1=Tx10
;  1-wire port line called as OWP1.
;  Calls TRESET1, TOUCHBY1

READ_T1:	MOV C,T1_PRS	;Present bit check
	JNC RT1		;Fail if no part on bus.
	LCALL TRESET1	;Issue reset pulse.
	JNC RT1		;Fail if no part on bus.
	MOV A,#0CCH	;Skip ROM command.
	LCALL TOUCHBY1	;Send command byte.
	MOV A,#0BEH	;Read scratchpad command.
	LCALL TOUCHBY1	;Send command byte.
	MOV A,#0FFH	;perform read
	LCALL TOUCHBY1	;Read LSB.
	MOV R2,A		;LSB store
	MOV A,#0FFH	;perform read
	LCALL TOUCHBY1	;Read MSB.
	MOV R3,A		;MSB store
	MOV A,#0FFH	;perform read
	LCALL TOUCHBY1	;Read TH.
	MOV A,#0FFH	;perform read
	LCALL TOUCHBY1	;Read TL.
	MOV A,#0FFH	;perform read
	LCALL TOUCHBY1	;Read RESERVED BYTE.
	MOV A,#0FFH	;perform read
	LCALL TOUCHBY1	;Read RESERVED BYTE.
	MOV A,#0FFH	;perform read
	LCALL TOUCHBY1	;Read COUNT REMAIN.
	MOV R4,A		;COUNT REMAIN store
	MOV A,#0FFH	;perform read
	LCALL TOUCHBY1	;Read COUNT PER C.
	MOV R5,A		;COUNT PER C store
	LCALL TCONV
	SETB C
RT1:	MOV T1_PR,C	;Make present bit
	RET

;  Read temperature from 1-wire thermometer DS1820.
;  Returns C=1 in case of valid code, else C=0.
;  Out   R2,R1=Tx10
;  1-wire port line called as OWP2.
;  Calls TRESET2, TOUCHBY2

READ_T2:	MOV C,T2_PRS	;Present bit check
	JNC RT2		;Fail if no part on bus.
	LCALL TRESET2	;Issue reset pulse.
	JNC RT2		;Fail if no part on bus.
	MOV A,#0CCH	;Skip ROM command.
	LCALL TOUCHBY2	;Send command byte.
	MOV A,#0BEH	;Read scratchpad command.
	LCALL TOUCHBY2	;Send command byte.
	MOV A,#0FFH	;perform read
	LCALL TOUCHBY2	;Read LSB.
	MOV R2,A		;LSB store
	MOV A,#0FFH	;perform read
	LCALL TOUCHBY2	;Read MSB.
	MOV R3,A		;MSB store
	MOV A,#0FFH	;perform read
	LCALL TOUCHBY2	;Read TH.
	MOV A,#0FFH	;perform read
	LCALL TOUCHBY2	;Read TL.
	MOV A,#0FFH	;perform read
	LCALL TOUCHBY2	;Read RESERVED BYTE.
	MOV A,#0FFH	;perform read
	LCALL TOUCHBY2	;Read RESERVED BYTE.
	MOV A,#0FFH	;perform read
	LCALL TOUCHBY2	;Read COUNT REMAIN.
	MOV R4,A		;COUNT REMAIN store
	MOV A,#0FFH	;perform read
	LCALL TOUCHBY2	;Read COUNT PER C.
	MOV R5,A		;COUNT PER C store
	LCALL TCONV
	SETB C
RT2:	MOV T2_PR,C	;Make present bit
	RET

;  Temperature code conversion
;  Truncates T<0
;  Input R3,R2=MSB,LSB of temperature
;  Input R4=COUNT REMAIN
;  Input R5=COUNT PER C
;  Out   R2,R1=Tx10

TCONV:    MOV A,R3		;MSB restore
	ANL A,#1H
	JNZ TMINUS

	MOV A,R2		;LSB restore
	CLR C
	RRC A
	MOV B,#10
	MUL AB		;B,A=Tx10

	ADD A,#7
	MOV R1,A
	MOV A,B
	ADDC A,#0
	MOV R2,A		;R2,R1=Tx10+7

	MOV A,R5
	MOV B,#10
	DIV AB		;A=COUNT PER C/10

	MOV B,A
	MOV A,R4
	DIV AB		;A=COUNT REMAIN/(COUNT PER C/10)

	XCH A,R1
	SUBB A,R1
	MOV R1,A
	MOV A,R2
	SUBB A,#0H
	MOV R2,A
	JC TMINUS		;R2,R1=T(actual)x10
	RET

TMINUS:	CLR A		;if T<0 then T=0
	MOV R2,A
	MOV R1,A
	RET

;Read temperature T1 and T2, get max, check for limit
;Uses: READ_T1, READ_T2
;Out:  TF_L, TF_R, FANF, FANH

READ_T:	MOV R2,#0H	;clear R2,R1 for zero ind. if th. not present
	MOV R1,#0H

	LCALL READ_T1	;read T1 and start next conversion
	MOV A,#HI(TFLIM)	;load HI(TFLIM)
	CJNE A,R02,TNE1	;check for fail limit
	MOV A,#LO(TFLIM)	;load LO(TFLIM)
	CJNE A,R01,TNE1
TNE1:	ORL C,TF_L
	MOV TF_L,C	;set TF_L if T1>TFLIM
	MOV TEMPH,R2	;save T1 value
	MOV TEMPL,R1

	LCALL READ_T2	;read T2 and start next conversion
	MOV A,#HI(TFLIM)	;load HI(TFLIM)
	CJNE A,R02,TNE2	;check for fail limit
	MOV A,#LO(TFLIM)	;load LO(TFLIM)
	CJNE A,R01,TNE2
TNE2:	ORL C,TF_R
	MOV TF_R,C	;set TF_R if T2>TFLIM
	MOV A,R2		;compare T1 and T2
	CJNE A,TEMPH,TNE12
	MOV A,R1
	CJNE A,TEMPL,TNE12
TNE12:	JC T1_MORE
	MOV TEMPH,R2	;save T2 value
	MOV TEMPL,R1

T1_MORE:  MOV A,#HI(TILIM)	;load HI(TILIM)
	CJNE A,TEMPH,TNELI	;check for indication limit
	MOV A,#LO(TILIM)	;load LO(TILIM)
	CJNE A,TEMPL,TNELI
TNELI:	JNC TNI
	MOV DISPRE,DISPMD	;set display pre-mode = display mode
	MOV A,#DM_G	;temperature display mode
	LCALL DSPMDA

TNI:	MOV A,#HI(THLIM)	;load HI(THLIM)
	CJNE A,TEMPH,TNELH	;check for limit
	MOV A,#LO(THLIM)	;load LO(THLIM)
	CJNE A,TEMPL,TNELH
TNELH:	MOV FANH,C	;set/clear fan hi-force bit

	MOV A,#HI(TLLIM)	;load HI(TLLIM)
	CJNE A,TEMPH,TNELL	;check for limit
	MOV A,#LO(TLLIM)	;load LO(TLLIM)
	CJNE A,TEMPL,TNELL
TNELL:	MOV FANF,C	;set/clear fan force bit
	RET

;Protection check
;In:  OKL, OKR, IOF, DCF
;Out: PF_L, DF_L, IF_L, PF_R, DF_R, IF_R

PR_CH:	JNB OKL,PR_CH1	;jump if left channel is OK (OKL=0)
	SETB PF_L		;set protection fail flag
	MOV C,DCF
	CPL C
	MOV DF_L,C
	MOV C,IOF
	CPL C
	MOV IF_L,C
PR_CH1:	JNB OKR,PR_CH2	;jump if OKR=0
	SETB PF_R		;set protection fail flag
	MOV C,DCF
	CPL C
	MOV DF_R,C
	MOV C,IOF
	CPL C
	MOV IF_R,C
PR_CH2:	RET

;  AMUX  CD4051B and ADC AD7896 support

;Ground ADC input

AMUX_G:	CLR AX0		;set AMUX address
	SETB AX1
	CLR AX2
	RET

;Sets AMUX channel for U1-U4 measure
;Input: U_CH = 0..255 (useful only bits 0,1):
;   U_CH=0 - +R (U1)
;   U_CH=1 - +L (U2)
;   U_CH=2 - -R (U3)
;   U_CH=3 - -L (U4)
;Out: AX0, AX1, AX2, U_CH = U_CH + 1

SET_UCH:	INC U_CH		;next U channel
	MOV A,U_CH
	ANL A,#03H
	JNZ SCH1		;jump if U_CH<>0
	CLR AX0		;set AMUX address
	CLR AX1
	SETB AX2
	RET

SCH1:	DEC A
	JNZ SCH2		;jump if U_CH<>1
	SETB AX0		;set AMUX address
	CLR AX1
	SETB AX2
	RET

SCH2:	DEC A
	JNZ SCH3		;jump if U_CH<>2
	CLR AX0		;set AMUX address
	SETB AX1
	SETB AX2
	RET

SCH3:	SETB AX0		;set AMUX address
	SETB AX1
	SETB AX2
	RET

;Converts in cycle U1-U4 (Call SET_UCH before ADC_UCH)
;U_CH.0=0 - right, U_CH.1=1 - left.
;U_CH.1=0 - pos. , U_CH.1=1 - neg.
;Out: UF_L, UF_R
;Uses ADC_U

ADC_UCH:	MOV R0,#10
	DJNZ R0,$		;delay 20uS for AMUX settling
	LCALL ADC_U	;start and read ADC
	MOV A,R3
#IFDEF PR_OFF
	RET
#ENDIF
	JNB U_CH.1,UPOS	;jump if U pos.
	CLR C		;invert U
	MOV A,#59
	SUBB A,R3

UPOS:     JB F_UOK,UPRO	;jump if F_UOK=1
	CJNE A,#UHLIM,$+3H	;compare U and UHLIM (C=1 if U<UHLIM)
	JB U_CH.0,LUFL	;jump if U_CH.0=1 (left channel)
	MOV UF_L,C
	SJMP UFR		;exit
LUFL:	MOV UF_R,C
	SJMP UFR		;exit

UPRO:	CJNE A,#ULLIM,$+3H	;compare U and ULLIM
	JNC UFR		;exit if U>ULLIM
	JB U_CH.0,LDFL	;jump if U_CH.0=1 (left channel)
	SETB UF_R
	SJMP UFR		;exit
LDFL:	SETB UF_L
UFR:	RET		;exit

;Start ADC and read U code
;Output: R3 - U code:  0,0,D11,D10,D9,D8,D7,D6
;Range 0..+64V - 0..64
;Range 0..-59V - 59..0

ADC_U:	CLR START		;start ADC
	SETB START

	MOV R0,#10
	DJNZ R0,$		;delay 20uS for ADC ready

	MOV R0,#2H	;bit counter load
ADCRDU1:	SETB SCLK
	MOV C,SDATA
	RLC A
	CLR SCLK
	DJNZ R0,ADCRDU1

	MOV R0,#8H	;bit counter load
ADCRDU2:	SETB SCLK
	MOV C,SDATA
	RLC A
	CLR SCLK
	DJNZ R0,ADCRDU2
	ANL A,#3FH
	MOV R3,A		;save code

	MOV R0,#6H	;bit counter load
ADCRDU3:	SETB SCLK
	MOV C,SDATA
	RLC A
	CLR SCLK
	DJNZ R0,ADCRDU3

	CLR AX0		;ground ADC input
	SETB AX1
	CLR AX2
	MOV R0,#10
	DJNZ R0,$		;delay 20uS

	RET

;ADC present detect
;Out: bit ADCPR = 1 if ADC present

ADC_DET:  CLR ADCPR
	LCALL AMUX_G
	LCALL ADC_U
	MOV A,R3
	ANL A,#0F0H
	JNZ ADR1
	SETB ADCPR
ADR1:	RET

;Sets AMUX channel and measure W and WP
;Input: ZELH,ZELL, ZERH,ZERL
;Out:   WH,WL, WPH,WPL

GET_POW:	CLR AX0		;set AMUX address for PWL
	CLR AX1
	CLR AX2

	MOV R0,#50
	DJNZ R0,$		;delay 100uS for AMUX settling

	LCALL ADC_PW	;start and read ADC

	CLR C
	MOV A,R3
	SUBB A,ZELL	;zero compensation
	MOV R1,A		;ADC code store
	MOV A,R4
	SUBB A,ZELH
	MOV R2,A
	JNC PW_CH_R
	MOV R2,#0H
	MOV R1,#0H

PW_CH_R:  SETB AX0		;set AMUX address for PWR
	CLR AX1
	CLR AX2

	MOV R0,#50
	DJNZ R0,$		;delay 100uS for AMUX settling

	LCALL ADC_PW	;start and read ADC

	CLR C
	MOV A,R3
	SUBB A,ZERL	;zero compensation
	MOV R3,A		;ADC code store
	MOV A,R4
	SUBB A,ZERH
	MOV R4,A
	JNC PW_MAX
	MOV R4,#0H
	MOV R3,#0H

PW_MAX:	MOV A,R4
	CJNE A,R02,GMAX	;get max from R4,R3 and R2,R1
	MOV A,R3
	CJNE A,R01,GMAX
GMAX:	JC PEAKPW		;jump if R2,R1 > R4,R3
	MOV R1,R03	;load R2,R1 from R4,R3
	MOV R2,R04

PEAKPW:   MOV A,R2		;get max from WPH,WPL and R2,R1
	CJNE A,WPH,GPMAX
	MOV A,R1
	CJNE A,WPL,GPMAX
GPMAX:	JC PWMAX		;jump if WPH,WPL > R2,R1
	MOV WPL,R1	;load WPH,WPL from R2,R1
	MOV WPH,R2
	MOV PHLDTM,#PHLDTMV	;load peak pover hold timer
	MOV DISPTM,#0	;display timer clear
	SJMP AVGPW
PWMAX:	MOV A,PHLDTM	;hold timer check
	JNZ AVGPW		;no timer overflow, no changes in WPH,WPL
	MOV A,PDECTM	;power dec timer check
	JNZ AVGPW		;no timer overflow, no changes in WPH,WPL
	MOV A,WPH
	SWAP A
	MOV R4,A
	ANL A,#00FH
	XCH A,R4
	ANL A,#0F0H
	MOV R3,A
	MOV A,WPL
	SWAP A
	ANL A,#00FH
	ORL A,R3
	MOV R3,A		;R4,R3 = WPH,WPL div 16
	ORL A,R4
	JNZ PCKDEC
	INC R3		;if R4,R3 = 0 then R3 = 1
PCKDEC:	CLR C
	MOV A,WPL
	SUBB A,R3
	MOV WPL,A
	MOV A,WPH
	SUBB A,R4
	MOV WPH,A		;WPH,WPL = WPH,WPL - WPH,WPL div 16
	JNC AVGPW
	MOV WPH,#0H
	MOV WPL,#0H

AVGPW:	MOV A,R2		;get max from WH,WL and R2,R1
	CJNE A,WH,GAMAX
	MOV A,R1
	CJNE A,WL,GAMAX
GAMAX:	JC PAVGD		;jump if WH,WL > R2,R1

	CLR C
	MOV A,WS
	SUBB A,WL
	MOV WS,A
	MOV A,WL
	SUBB A,WH
	MOV WL,A
	MOV A,WH
	SUBB A,#0H
	MOV WH,A		;WH,WL,WS - WH,WL,SW/256

	MOV A,WS
	ADD A,R1
	MOV WS,A
	MOV A,WL
	ADDC A,R2
	MOV WL,A
	MOV A,WH
	ADDC A,#0H
	MOV WH,A		;WH,WL,WS + R2,R1
	MOV A,PDECTM	;power dec timer check
	JNZ AVG_R		;no timer overflow, ret
	MOV PDECTM,#PDRATE	;power dec timer reload
	SJMP AVG_R

PAVGD:	MOV A,PDECTM	;power dec timer check
	JNZ AVG_R		;no timer overflow, no changes in WH,WL
	MOV PDECTM,#PDRATE	;power dec timer reload
	MOV A,WH
	SWAP A
	MOV R4,A
	ANL A,#00FH
	XCH A,R4
	ANL A,#0F0H
	MOV R3,A
	MOV A,WL
	SWAP A
	ANL A,#00FH
	ORL A,R3
	MOV R3,A		;R4,R3 = WH,WL div 16
	ORL A,R4
	JNZ AVGDEC
	INC R3		;if R4,R3 = 0 then R3 = 1
AVGDEC:	CLR C		;WH,WL = WH,WL - STEPD
	MOV A,WL
	SUBB A,R3
	MOV WL,A
	MOV A,WH
	SUBB A,R4
	MOV WH,A		;WH,WL = WH,WL - WH,WL div 16
	JNC AVG_R
	MOV WS,#0H
	MOV WH,#0H
	MOV WL,#0H
AVG_R:	RET

;Start ADC and read PW code
;Output: R3 - low  byte:  D7,D6,D5,D4,D3,D2,D1,D0
;        R4 - high byte:  0,0,0,0,D11,D10,D9,D8

ADC_PW:	CLR START		;start ADC
	SETB START

	MOV R0,#10
	DJNZ R0,$		;delay 20uS for ADC ready

	MOV R0,#8H	;bit counter load
ADC_PW1:	SETB SCLK
	MOV C,SDATA
	RLC A
	CLR SCLK
	DJNZ R0,ADC_PW1
	ANL A,#0FH
	MOV R4,A		;save high byte

	MOV R0,#8H	;bit counter load
ADC_PW2:	SETB SCLK
	MOV C,SDATA
	RLC A
	CLR SCLK
	DJNZ R0,ADC_PW2
	MOV R3,A		;save low byte

	CLR AX0		;ground ADC input
	SETB AX1
	CLR AX2
	MOV R0,#10
	DJNZ R0,$		;delay 20uS

	RET

;ADC zero calibration
;Input: ZCAL=16, ZELH,ZELL=0, ZERH,ZERL=0
;Out after 16 calls:
;ZCAL=0,
;ZELH,ZELL=zero error left,
;ZERH,ZERL=zero error right
;Uses:  ADC_PW

ZCALIB:	CLR AX0		;set AMUX address for PWL
	CLR AX1
	CLR AX2

	MOV R0,#50
	DJNZ R0,$		;delay 100uS for AMUX settling

	LCALL ADC_PW	;start and read ADC
	MOV A,ZELL	;sum R4,R3 and ZELH,ZELL
	ADD A,R3
	MOV ZELL,A
	MOV A,ZELH
	ADDC A,R4
	MOV ZELH,A

	SETB AX0		;set AMUX address for PWR
	CLR AX1
	CLR AX2

	MOV R0,#50
	DJNZ R0,$		;delay 100uS for AMUX settling

	LCALL ADC_PW	;start and read ADC
	MOV A,ZERL	;sum R4,R3 and ZERH,ZERL
	ADD A,R3
	MOV ZERL,A
	MOV A,ZERH
	ADDC A,R4
	MOV ZERH,A

	DEC ZCAL
	MOV A,ZCAL
	JNZ CALB

	MOV R2,ZELH
	MOV R1,ZELL
	MOV R0,#4H
SHLZ:	LCALL SHIFT
	DJNZ R0,SHLZ
	MOV ZELH,R2
	MOV ZELL,R1

	MOV R2,ZERH
	MOV R1,ZERL
	MOV R0,#4H
SHRZ:	LCALL SHIFT
	DJNZ R0,SHRZ
	MOV ZERH,R2
	MOV ZERL,R1

CALB:	RET

;Clear non-display LEDs

LEDCLR:   CLR LEDTMR
	CLR LEDEXT
	CLR LEDMNP
	CLR LEDOPR
	CLR LEDRFL
	CLR LEDLDR
	CLR LEDLFL
	RET

;Indication clear
;Clears hardware registers REG1-REG5

INDCLR:	CLR A
	CPL A
	MOV R0,#REG1
	MOVX @R0,A
	MOV R0,#REG2
	MOVX @R0,A
	MOV R0,#REG3
	MOVX @R0,A
	MOV R0,#REG4
	MOVX @R0,A
	MOV R0,#REG5
	MOVX @R0,A
	RET

;Copy WPH, WPL, WH, WL to WPHC, WPLC, WHC, WLC

COPY:	MOV WPHC,WPH
	MOV WPLC,WPL
	MOV WHC,WH
	MOV WLC,WL
	RET

;Display power (W), peak power (WP), temperature (G), timer (M), fail (L)
;Input: DISPMD, WHC,WLC or WPHC,WPLC or TEMPH,TEMPL or OFFTM, or DIG1-DIG4

DISP:	MOV A,DISPMD
	CJNE A,#DM_W,IND1	;if DISPMD = DM_W then display power
	MOV A,NOW_MD
	CJNE A,#FL_MD,IND_W
	LJMP IND_FL
IND_W:	MOV R2,WHC
	MOV R1,WLC
	LCALL SHIFT	;shift R2,R1 (/2)
	LCALL B2BCD
	JB ADCPR,APR1
	MOV A,#CH_MIN
	MOV DIG1,A
	MOV DIG2,A
	MOV DIG3,A
	MOV DIG4,A
APR1:	LCALL IND
	RET

IND1:	CJNE A,#DM_WP,IND2	;if DISPMD = DM_WP then display peak power
	MOV A,NOW_MD
	CJNE A,#FL_MD,IND_WP
	LJMP IND_FL
IND_WP:	MOV R2,WPHC
	MOV R1,WPLC
	LCALL SHIFT	;shift R2,R1 (/2)
	LCALL B2BCD
	JB ADCPR,APR2
	MOV A,#CH_MIN
	MOV DIG1,A
	MOV DIG2,A
	MOV DIG3,A
	MOV DIG4,A
APR2:	LCALL IND
	RET

IND2:	CJNE A,#DM_G,IND3	;if DISPMD = DM_G then display temperature
	MOV R2,TEMPH
	MOV R1,TEMPL
	LCALL B2BCD
	MOV C,T1_PR
	ORL C,T2_PR
	JC TPR2
	MOV A,#CH_MIN
	MOV DIG1,#BLANK
	MOV DIG2,A
	MOV DIG3,A
	MOV DIG4,A
TPR2:	LCALL IND
	RET

IND3:	CJNE A,#DM_M,IND4	;if DISPMD = DM_M then display timer
	MOV A,OFFTM
	JNZ IND31
	MOV DIG1,#BLANK
	MOV DIG2,#CH_O	;display "OFF" if OFFTM = 0
	MOV DIG3,#CH_F
	MOV DIG4,#CH_F
	LCALL IND
	RET
IND31:    MOV R2,#0H
	MOV R1,A
	LCALL B2BCD
	LCALL IND
	RET

IND4:	CJNE A,#DM_F,IND5	;if DISPMD = DM_F then display off
	MOV DIG1,#BLANK
	MOV DIG2,#BLANK
	MOV DIG3,#BLANK
	MOV DIG4,#BLANK
	LCALL IND

IND5:	RET

IND_FL:	CLR POINT
	MOV DIG1,#BLANK
	MOV DIG4,#CH_F

	MOV C,IF_L
	ORL C,IF_R
	JNC SFLI1
	MOV DIG2,#BLANK
	MOV DIG3,#CH_I
	LCALL IND		;display IF if IF=1
	RET

SFLI1:	MOV C,DF_L
	ORL C,DF_R
	JNC SFLI2
	MOV DIG2,#CH_d
	MOV DIG3,#CH_c
	LCALL IND		;display dcF if DF=1
	RET

SFLI2:	MOV C,UF_L
	ORL C,UF_R
	JNC SFLI3
	MOV DIG2,#BLANK
	MOV DIG3,#CH_U
	LCALL IND		;display UF if UF=1
	RET

SFLI3:	MOV C,PF_L
	ORL C,PF_R
	JNC SFLI4
	MOV DIG2,#CH_P
	MOV DIG3,#CH_r
	LCALL IND		;display PrF if PF=1
	RET

SFLI4:	MOV C,TF_L
	ORL C,TF_R
	JNC SFLI5
	MOV DIG2,#BLANK
	MOV DIG3,#CH_t
	LCALL IND		;display tF if TF=1
	RET

SFLI5:	MOV DIG2,#CH_E
	MOV DIG3,#CH_r
	LCALL IND		;display ErF if program error
	RET

;Shift R2,R1 (/2)
;Input: R2,R1
;Out: R2,R1 = R2,R1 / 2

SHIFT:    CLR C
	MOV A,R2
	RRC A
	MOV R2,A
	MOV A,R1
	RRC A
	MOV R1,A
	RET

;Convert R2,R1 in binary format to BCD
;Out: DIG1..4 = 0000 - 9999

B2BCD:    MOV DIG1,#0FFH	;1000s init

DLC1:	MOV R4,R02
	MOV R3,R01
	INC DIG1
	CLR C
	MOV A,R3
	SUBB A,#0E8H
	MOV R1,A
	MOV A,R4
	SUBB A,#003H
	MOV R2,A
	JNC DLC1		;if C=1 then DIG1=N/1000

	MOV R2,R04
	MOV R1,R03
	MOV DIG2,#0FFH	;100s init

	MOV R0,#0H
	MOV A,DIG1
	JNZ DLC2
	MOV DIG1,#BLANK	;zero supression
	MOV R0,#BLANK	;save leading zero flag

DLC2:	MOV R4,R02
	MOV R3,R01
	INC DIG2
	CLR C
	MOV A,R3
	SUBB A,#100
	MOV R1,A
	MOV A,R4
	SUBB A,#0
	MOV R2,A
	JNC DLC2		;if C=1 then DIG1=N/100,R4=0

	MOV A,DIG2
	JNZ DLC3
	MOV DIG2,R0	;zero supression

DLC3:	MOV A,R3
	MOV B,#10
	DIV AB		;A=N/10,B=N/1
	MOV DIG3,A
	MOV DIG4,B

	RET

;Indication
;Input: DIG1-DIG4 (character code), LEDs bits
;Loads hardware registers REG1-REG5

IND:	MOV A,NOW_MD
	XRL A,#FL_MD
	ADD A,#0FFH	;C = 0 if NOW_MD = #FL_MD
	MOV F0,C

	MOV A,FAIL_L
	ADD A,#0FFH	;C = 0 if FAIL_L = 0
	ANL C,BLINK
	ANL C,/F0
	MOV LEDLFL,C	;fail left  led control

	MOV A,FAIL_R
	ADD A,#0FFH	;C = 0 if FAIL_R = 0
	ANL C,BLINK
	ANL C,/F0
	MOV LEDRFL,C	;fail right led control

IND_NF:	MOV DPTR,#FONT

	MOV A,DIG1
	MOVC A,@A+DPTR
	MOV C,LEDEXT
	CPL C
	MOV ACC.0,C
	MOV R0,#REG1
	MOVX @R0,A

	MOV A,DIG2
	MOVC A,@A+DPTR
	MOV C,LEDDGR
	CPL C
	MOV ACC.0,C
	MOV R0,#REG2
	MOVX @R0,A

	MOV A,DIG3
	MOVC A,@A+DPTR
	MOV C,POINT
	CPL C
	MOV ACC.0,C
	MOV R0,#REG3
	MOVX @R0,A

	MOV A,DIG4
	MOVC A,@A+DPTR
	MOV C,LEDWTT
	CPL C
	MOV ACC.0,C
	MOV R0,#REG4
	MOVX @R0,A

	MOV A,LEDS2
	CPL A
	MOV R0,#REG5
	MOVX @R0,A

	RET

;  ------ Interrupts Holders ------

;  --- TIMER 0 Interrupt ---
;  System clock 20mS

RTC:	PUSH PSW
	PUSH ACC
	CLR TR0          	;timer 0 stop
	MOV TH0,#HI(RTCV)  	;timer 0 load for 20 mS
	MOV TL0,#LO(RTCV)
	SETB TR0         	;timer start

	INC RTPC         	;Real Time Program Counter INC
	DJNZ RTPCS,RTC1
	MOV RTPCS,#V1S

	DJNZ RTPCM,RTC10
	MOV RTPCM,#V1M

;  1MIN program counters

	MOV A,OFFTM
	JZ RTC20
	DEC OFFTM

RTC20:	MOV A,AUFTM
	JZ RTC10
	DEC AUFTM

;  1S program counters

RTC10:	MOV A,FANTM
	JZ RTC11
	DEC FANTM

RTC11:	MOV A,UWTTM
	JZ RTC1
	DEC UWTTM

;  20mS program counters

RTC1:	MOV A,KEYHTM
	JZ RTC2
	DEC KEYHTM

RTC2:	MOV A,TEMPTM
	JZ RTC3
	DEC TEMPTM

RTC3:	MOV A,DISPTM
	JZ RTC4
	DEC DISPTM

RTC4:	MOV A,DELTM
	JZ RTC5
	DEC DELTM

RTC5:	MOV A,PHLDTM
	JZ RTC6
	DEC PHLDTM

RTC6:	MOV A,PDECTM
	JZ RTC7
	DEC PDECTM

RTC7:	POP ACC
	POP PSW
	RETI

;  --- TIMER 1 Interrupt ---
;  Sound generation

SND:      PUSH ACC
	MOV TL1,FREQL
	MOV TH1,FREQH
	CPL BPR
	DEC SDUR
	MOV A,SDUR
	JNZ SND1
	SETB BPR
	CLR TR1		;timer 1 stop
	SETB EX1		;keyboard interrupt enable
	SETB EX1_EN	;keyboard interrupt enable flag set
SND1:     POP ACC
	RETI

;  --- INT1 Interrupt ---

;  RC5 program decoder is interrupt holder.
;  Input - bit SER (interrupt line), low active level.
;  Out - byte R37 (register R7 in bank 3),
;                  D0-D5 - command bits
;                  D6 - control bit of local keyboard
;                 (D6 does not change)
;                  D7 - control bit RC
;  Uses register bank 3

RC5_DF	.EQU B.0		;data flag
RC5_PF	.EQU B.1		;period flag
RC5_N	=(2*(CLK_KHZ/188)+1)/2
RC5_N1	=(2*(CLK_KHZ/249)+1)/2
RC5_N2	=(2*(CLK_KHZ/382)+1)/2
RC5_NT	=(2*(CLK_KHZ/135)+1)/2

RC5:      PUSH PSW		;RC5-code program decoding
	PUSH ACC
	PUSH B
	SETB RS0
	SETB RS1		;bank 3
	MOV R6,DPH
	MOV R5,DPL

	MOV B,#2
	MOV R2,#2		;system byte init for 7 received bits
	MOV R3,#4		;command byte init for 6 received bits
	MOV R0,#R32	;R32 address load

	MOV R1,#RC5_N
WLOW:	JB SER,WHIGH
	MOV R4,#16
	DJNZ R4,$
	DJNZ R1,WLOW	;wait up to 2362uS for SER = 1
	SJMP NORC5	;jump if time is over

WHIGH:	MOV R1,#RC5_N
SAMPLE:	MOV R4,#8
	DJNZ R4,$
SAM1:	MOV R4,#3
	DJNZ R4,$
SAM2:	MOV R4,#2
	DJNZ R4,$
SAM3:	MOV C,SER
	MOV ACC.0,C
	XRL A,B
	JNB ACC.0,TRANS
	DJNZ R1,SAMPLE	;sample SER up to 2362uS
	SJMP NORC5	;jump if time is over

TRANS:	CPL RC5_DF	;transition detected
	MOV A,R1
	ADD A,#-RC5_N1
	JC NORC5		;jump if <577.5uS
	MOV A,R1
	ADD A,#-RC5_N2
	JC TRANS1
	JB RC5_PF,NORC5	;jump if >1200uS
	SJMP STDATA
TRANS1:	CPL RC5_PF
	MOV R1,#RC5_N
	JB RC5_PF,SAM1
STDATA:	MOV C,RC5_DF
	MOV A,@R0
	RLC A		;save data bit
	MOV @R0,A
	MOV R1,#RC5_N
	JNC SAM2
	INC R0		;system done, begin command
	MOV A,R0
	MOV R1,#RC5_N
	JNB ACC.2,SAM3

	MOV R1,#RC5_N
WLW:	JB SER,TRM
	MOV R4,#16
	DJNZ R4,$
	DJNZ R1,WLW	;wait up to 2362uS for SER = 1
	SJMP NORC5	;jump if time is over

TRM:	MOV R1,#RC5_NT
TERM:	JNB SER,NORC5
	MOV R4,#16
	DJNZ R4,$
	DJNZ R1,TERM

	MOV A,R2		;system nom. and control bit in R2
	ANL A,#1FH
	CJNE A,#SYS,NORC5	;system number check

	MOV DPTR,#RCTAB	;table address load
	MOV A,R3		;command code load
	MOVC A,@A+DPTR	;recoding
	MOV R3,A		;new code store

	MOV A,R2		;system nom. and control bit in R2
	ANL A,#20H	;control bit separating
	RL A
	RL A		;D7-control bit RC
	ORL A,R3		;combine command code and control bit
	MOV R3,A		;store

	MOV A,R7
	ANL A,#40H	;save D6-control bit of local keyboard
	ORL A,R3		;combine command code and D6,D7
	MOV R7,A		;store command data

NORC5:
#IF SER=INT0		;interrupt flag clear
	CLR IE0
#ELSE
	CLR IE1
#ENDIF
	MOV DPL,R5
	MOV DPH,R6
	POP B
	POP ACC
	POP PSW
	RETI

;RC Recoding Table

RCTAB     .DB NONE		;key code 000H, key function - none
	.DB NONE		;key code 001H, key function - none
	.DB NONE		;key code 002H, key function - none
	.DB NONE		;key code 003H, key function - none
	.DB NONE		;key code 004H, key function - none
	.DB NONE		;key code 005H, key function - none
	.DB NONE		;key code 006H, key function - none
	.DB NONE		;key code 007H, key function - none
	.DB NONE		;key code 008H, key function - none
	.DB NONE		;key code 009H, key function - none
	.DB NONE 		;key code 00AH, key function - none
	.DB NONE		;key code 00BH, key function - none
	.DB CODSTB	;key code 00CH, key function - STANDBY
	.DB NONE		;key code 00DH, key function - none
	.DB NONE		;key code 00EH, key function - none
	.DB NONE		;key code 00FH, key function - none
	.DB NONE 		;key code 010H, key function - none
	.DB NONE 		;key code 011H, key function - none
	.DB NONE		;key code 012H, key function - none
	.DB NONE		;key code 013H, key function - none
	.DB NONE		;key code 014H, key function - none
	.DB NONE		;key code 015H, key function - none
	.DB NONE		;key code 016H, key function - none
	.DB NONE		;key code 017H, key function - none
	.DB NONE		;key code 018H, key function - none
	.DB NONE		;key code 019H, key function - none
	.DB NONE		;key code 01AH, key function - none
	.DB NONE		;key code 01BH, key function - none
	.DB NONE		;key code 01CH, key function - none
	.DB NONE		;key code 01DH, key function - none
	.DB NONE 		;key code 01EH, key function - none
	.DB NONE		;key code 01FH, key function - none
	.DB CODMOD	;key code 020H, key function - MODE
	.DB CODDIS	;key code 021H, key function - DISPLAY
	.DB NONE 		;key code 022H, key function - none
	.DB NONE		;key code 023H, key function - none
	.DB NONE		;key code 024H, key function - none
	.DB NONE		;key code 025H, key function - none
	.DB NONE		;key code 026H, key function - none
	.DB NONE		;key code 027H, key function - none
	.DB NONE		;key code 028H, key function - none
	.DB NONE		;key code 029H, key function - none
	.DB NONE 		;key code 02AH, key function - none
	.DB NONE		;key code 02BH, key function - none
	.DB NONE		;key code 02CH, key function - none
	.DB NONE 		;key code 02DH, key function - none
	.DB NONE		;key code 02EH, key function - none
	.DB NONE		;key code 02FH, key function - none
	.DB NONE		;key code 030H, key function - none
	.DB NONE		;key code 031H, key function - none
	.DB NONE		;key code 032H, key function - none
	.DB NONE		;key code 033H, key function - none
	.DB NONE		;key code 034H, key function - none
	.DB NONE 		;key code 035H, key function - none
	.DB NONE		;key code 036H, key function - none
	.DB NONE 		;key code 037H, key function - none
	.DB NONE 		;key code 038H, key function - none
	.DB NONE		;key code 039H, key function - none
	.DB NONE		;key code 03AH, key function - none
	.DB NONE 		;key code 03BH, key function - none
	.DB NONE 		;key code 03CH, key function - none
	.DB NONE		;key code 03DH, key function - none
	.DB NONE		;key code 03EH, key function - none
	.DB NONE 		;key code 03FH, key function - none

;Local Keyboard Recoding Table

KEYTAB:   .DB CODTIM	;key code 000H, key function - TIMER
	.DB CODPKP	;key code 001H, key function - PEAK
	.DB CODMIN	;key code 002H, key function - MINUS
	.DB CODSTB	;key code 003H, key function - STANDBY
	.DB CODTMP	;key code 004H, key function - TEMP
	.DB CODPLS	;key code 005H, key function - PLUS
	.DB NONE		;key code 006H, key function - none
	.DB NONE		;key code 007H, key function - none
	.DB NONE		;key code 008H, key function - none

; ------ Control Functions Codes ------

NONE	.EQU 000H	;key function - none
CODSTB	.EQU 001H	;key function - STANDBY
CODDIS	.EQU 002H	;key function - DISPLAY
CODMOD	.EQU 003H	;key function - MODE
CODPKP	.EQU 004H	;key function - PEAK
CODTIM	.EQU 005H	;key function - TIMER
CODTMP	.EQU 006H	;key function - TEMP
CODMIN	.EQU 007H ;key function - MINUS
CODPLS	.EQU 008H	;key function - PLUS

;Font table
;	    CDBFGEAS
FONT	.DB 00001001B	;code 00H, character 0
	.DB 01011111B	;code 01H, character 1
	.DB 10010001B	;code 02H, character 2
	.DB 00010101B	;code 03H, character 3
	.DB 01000111B	;code 04H, character 4
	.DB 00100101B	;code 05H, character 5
	.DB 00100001B	;code 06H, character 6
	.DB 01011101B	;code 07H, character 7
	.DB 00000001B	;code 08H, character 8
	.DB 00000101B	;code 09H, character 9
	.DB 01000001B	;code 0AH, character A
	.DB 00100011B	;code 0BH, character B
	.DB 10101001B	;code 0CH, character C
	.DB 00010011B	;code 0DH, character D
	.DB 10100001B	;code 0EH, character E
	.DB 11100001B	;code 0FH, character F
	.DB 00001011B	;code 10H, character U
	.DB 01011111B	;code 11H, character I
	.DB 11000001B	;code 12H, character P
	.DB 00010011B	;code 13H, character d
	.DB 10110011B	;code 14H, character c
	.DB 10100011B	;code 15H, character t
	.DB 11110011B	;code 16H, character r
	.DB 11110111B	;code 17H, character -
	.DB 11111111B	;code 18H, character blank

; ------ Characters codes table ------

CH_A	.EQU 00AH		;character "A" code
CH_E	.EQU 00EH		;character "E" code
CH_O	.EQU 000H		;character "O" code
CH_F	.EQU 00FH		;character "F" code
CH_U	.EQU 010H		;character "U" code
CH_I	.EQU 011H		;character "I" code
CH_P	.EQU 012H		;character "P" code
CH_d	.EQU 013H		;character "d" code
CH_c	.EQU 014H		;character "c" code
CH_t	.EQU 015H		;character "t" code
CH_r	.EQU 016H		;character "r" code
CH_MIN	.EQU 017H		;character "-" code
BLANK	.EQU 018H		;character "blank" code

; ------ Off timer values table ------

OFFTBL	.DB 5
	.DB 15
	.DB 30
	.DB 45
	.DB 60
	.DB 90
	.DB 120
	.DB 0

	.END
