;  ------ NSM PA-2020 Power Amplifier Controller ------
;  ------ Processor: ATMEL AT89C2051 PC24 ------

#INCLUDE "LIBREG.ASM"	;8051 SFR set

CLK_KHZ	= 12000		;clock frequency
RTC_MS	= 20		;real time clock period

RTCV	= CLK_KHZ*RTC_MS
RTCV	= RTCV/12
RTCV	= - RTCV
RTCV	= RTCV & 0FFFFH
RTCVL	= RTCV & 0FFH
RTCVH	= RTCV >> 8
RTCVH	= RTCVH & 0FFH

;  ------ Constantes ------

STACK	.EQU 056H		;stack location
V1S	.EQU 50		;program timer RTPCS value 1S/20mS=50
DEL_ON	.EQU 50		;x20mS operate on delay value
DEL_RR	.EQU 100		;x20mS R1-R2 delay value
DEL_TO	.EQU 200		;x20mS voltage settling timeout
DEL_LD	.EQU 150		;x20mS load on delay
DEL_OF	.EQU 50		;x20mS load off delay

;  ------ Ports ------

REM	.EQU INT0		;Remote control line
REL2	.EQU INT1		;Relay 2
LEDR	.EQU T0		;LED red
LEDG	.EQU T1		;LED green
TRL	.EQU P1.2		;thermometer left
IOF	.EQU P1.3		;output stage current fail
ENB	.EQU P1.4		;enable
TRR	.EQU P1.5		;thermometer right
OKR	.EQU P1.6		;hardware protection right OK
OKL	.EQU P1.7		;hardware protection left  OK
WD	.EQU RXD		;Watchdog wakeup line
REL1	.EQU TXD		;Relay 1
TRHL	.EQU P3.7		;Comparator threshold
COMP	.EQU P3.6		;Comparator Out

;  ------ Variables ------

;  Bit addressing memory

RTPC	.EQU 020H		;Real time program counter
T040M	.EQU M_20H.0	;40mS  period bit
T080M	.EQU M_20H.1	;80mS  period bit
T160M	.EQU M_20H.2	;160mS period bit
T320M	.EQU M_20H.3	;320mS period bit
T640M	.EQU M_20H.4	;640mS period bit
T1S28 	.EQU M_20H.5	;1.28S period bit
T2S56	.EQU M_20H.6	;2.56S period bit
T5S12	.EQU M_20H.7	;5.12S period bit

LEDS	.EQU 021H
LEDS_OFF	.EQU 000H
LEDG_BLB	.EQU M_21H.0	;LED green blink bit
LEDG_BL	.EQU 001H
LEDG_ONB	.EQU M_21H.1	;LED green on bit
LEDG_ON	.EQU 002H
LEDR_BLB	.EQU M_21H.2	;LED red blink bit
LEDR_BL	.EQU 004H
LEDR_ONB	.EQU M_21H.3	;LED red on bit
LEDR_ON	.EQU 008H
LEDY_BL	.EQU 005H

PROC	.EQU 022H		;processing function code
MODE	.EQU 023H		;current mode code
COD_NO	.EQU 000H
COD_OP	.EQU 001H		;operate code
COD_ST	.EQU 002H		;standby code
COD_PF	.EQU 003H		;protection fail code

PHASE	.EQU 024H		;Standby/Operate phase

;  Internal data memory

OPSTM	.EQU 030H

;  ------ Vectors Area ------

	.ORG 0000H	;reset vector
	LJMP INIT
	.ORG 000BH	;INT TIMER 0 vector
	LJMP RTC

;  ------ Main Program ------

INIT:	MOV SP,#STACK
	MOV TMOD,#11H    	;timer 0 and timer 1 init
	CLR TR0          	;timer 0 stop
	CLR TR1          	;timer 1 stop
	MOV TL0,#RTCVL   	;timer 0 load
	MOV TH0,#RTCVH
	MOV RTPC,#0H     	;program counter load

	CLR PT0          	;timer 0 int low priority
	SETB ET0         	;timer 0 int enable
	SETB TR0         	;timer 0 start
	SETB EA          	;interrupts enable

	MOV PROC,#COD_NO	;no processing function
	MOV MODE,#COD_ST	;current function standby
	MOV LEDS,#LEDG_ON	;LED green on
	SETB TRHL		;set high voltage threshold

MAIN:     MOV A,MODE
	CJNE A,#COD_ST,MA1

;Standby mode:

	SJMP MA3

MA1:	CJNE A,#COD_OP,MA2

;Operation mode:

	MOV C,TRL		;test left heatsink temperature
	ANL C,TRR		;test right heatsink temperature
	ANL C,COMP	;test supply voltage
	ANL C,/OKL	;test left hardware protection
	ANL C,/OKR	;test right hardware protection
	JC MA3		;jump if OK
	LCALL PROT	;enter protection fail mode
	SJMP MA4

MA2:	CJNE A,#COD_PF,MA3

;Pritection fail mode:

	MOV C,TRL		;test left heatsink temperature
	ANL C,TRR		;test right heatsink temperature
	ANL C,IOF		;test output stage current fail
	JNC MA4		;jump if fail
	MOV LEDS,#LEDY_BL	;LED yellow blink
	SJMP MA4

MA3:	JB REM,REM1	;jump to operate
	LCALL STANDBY
	SJMP MA4
REM1:	LCALL OPERATE

MA4:	MOV C,T320M	;LED green control
	ANL C,LEDG_BLB
	ORL C,LEDG_ONB
	CPL C
	MOV LEDG,C

	MOV C,T320M	;LED red control
	ANL C,LEDR_BLB
	ORL C,LEDR_ONB
	CPL C
	MOV LEDR,C

	MOV C,T080M	;watchdog wake up
	MOV WD,C
	LJMP MAIN

;  ------ Subroutines Area ------

;  Operate function

OPERATE:  MOV A,PROC	;process code load
	XRL A,#COD_OP
	JZ PH0
	MOV PROC,#COD_OP
	MOV MODE,#COD_NO
	MOV PHASE,#0H	;process init

PH0:	MOV A,PHASE
	JNZ PH1
	MOV OPSTM,#DEL_ON	;operate on delay value
	INC PHASE		;next phase (1)
	RET

PH1:	DEC A
	JNZ PH2
	MOV A,OPSTM
	JZ PH11		;timer check
	RET
PH11:	INC PHASE		;next phase (2) if timer overflow
	RET

PH2:	DEC A
	JNZ PH3
	CLR REL1		;relay 1 on
	MOV LEDS,#LEDG_BL	;LED green blink
	MOV OPSTM,#DEL_TO	;timeout delay value
	INC PHASE		;next phase (3)
	RET

PH3:      DEC A
	JNZ PH4
	MOV A,OPSTM
	JZ ERR1		;check timer for timeout
	JB COMP,PH31	;jump if voltage OK
	RET
PH31:	INC PHASE		;next phase (4) if voltage OK
	CLR TRHL		;set low voltage threshold
	MOV OPSTM,#DEL_RR	;load R1-R2 delay value
	RET
ERR1:     SETB REL1		;relay 1 off
	SETB TRHL		;set high voltage threshold
	MOV LEDS,#LEDR_BL	;LED red blink
	MOV PHASE,#0FFH	;end of function (FF)
	RET

PH4:      DEC A
	JNZ PH5
	MOV A,OPSTM
	JZ PH41		;check timer
	RET
PH41:	JNB COMP,ERR2	;jump if voltage OK
	JB OKL,ERR2
	JB OKR,ERR2
	CLR REL2		;relay 2 on
	MOV OPSTM,#DEL_LD	;load on delay value
	INC PHASE		;next phase (5)
	RET
ERR2:	SETB REL1		;relay 1 off
	SETB TRHL		;set high voltage threshold
	MOV LEDS,#LEDR_BL	;LED red blink
	MOV PHASE,#0FFH	;end of function (FF)
	RET

PH5:      DEC A
	JNZ PH6
	MOV A,OPSTM
	JZ PH51		;timer check
	RET
PH51:     JB OKL,ERR3
	JB OKR,ERR3
	CLR ENB		;relay 3 and 4 enable
	MOV MODE,#COD_OP
	MOV LEDS,#LEDR_ON	;LED red on
	INC PHASE		;next phase (6)
	RET
ERR3:     SETB REL2		;relay 2 off
	SETB REL1		;relay 1 off
	SETB TRHL		;set high voltage threshold
	MOV LEDS,#LEDR_BL	;LED red blink
	MOV PHASE,#0FFH	;end of function (FF)
	RET

PH6:	RET

;  Standby function

STANDBY:  MOV A,PROC	;process code load
	XRL A,#COD_ST
	JZ PL0
	MOV PROC,#COD_ST
	MOV MODE,#COD_NO
	MOV PHASE,#0H	;process init

PL0:	MOV A,PHASE
	JNZ PL1
	JB ENB,PL01
	SETB ENB		;relay 3 and 4 disable
	MOV LEDS,#LEDG_BL
	MOV OPSTM,#DEL_OF	;operate off delay
PL01:	INC PHASE		;next phase (1)
	RET

PL1:	DEC A
	JNZ PL2
	MOV A,OPSTM
	JNZ PL11		;timer check
	INC PHASE		;next phase (2) if timer overflow
PL11:	RET

PL2:	DEC A
	JNZ PL3
	SETB REL1		;relay 1 off
	SETB REL2		;relay 2 off
	SETB TRHL		;set high voltage threshold
	MOV MODE,#COD_ST
	MOV LEDS,#LEDG_ON	;LED green on
	INC PHASE		;next phase (3)
	RET

PL3:      RET

;  Protection function

PROT:	SETB ENB		;relay 3 and 4 disable
	SETB REL1		;relay 1 off
	SETB REL2		;relay 2 off
	MOV LEDS,#LEDR_BL
	MOV PROC,#COD_NO	;no processing function
	MOV MODE,#COD_PF	;current mode - protection fail
	RET

;  ------ Interrupt Holders ------

;  TIMER 0 Interrupt
;  System clock 20mS

RTC:	PUSH PSW
	PUSH ACC
	CLR TR0          	;timer 0 stop
	MOV TH0,#RTCVH   	;timer 0 load for 20 mS
	MOV TL0,#RTCVL
	SETB TR0         	;timer start

	INC RTPC         	;Real Time Program Counter INC

;  20mS program counters

RTC1:	MOV A,OPSTM
	JZ RTC2
	DEC OPSTM

RTC2:     POP ACC
	POP PSW
	RETI

	.END
